<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestBody;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use EshopOrders\AdminModule\Model\Customers;
use EshopOrders\ApiModule\Api\V1\Model\Customers as ApiCustomers;
use Users\Model\Security\User;

/**
 * Class CustomersController
 * @Path("/customers")
 */
class CustomersController extends BaseController
{
	public function __construct(
		protected User         $user,
		protected Customers    $customersService,
		protected ApiCustomers $apiCustomers,
	)
	{
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$qb        = $this->customersService->getEr()->createQueryBuilder('p')
			->addSelect('user, adDel, adInv, grp')
			->innerJoin('p.user', 'user')
			->leftJoin('p.addressDelivery', 'adDel')
			->leftJoin('p.addressInvoice', 'adInv')
			->leftJoin('p.groupCustomers', 'grp');
		$customers = $qb->getQuery()->getResult();

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($customers));
	}

	/**
	 * @Path("/in-group")
	 * @Method("GET")
	 */
	public function inGroup(ApiRequest $request, ApiResponse $response)
	{
		$qb        = $this->customersService->getEr()->createQueryBuilder('p')
			->select('user.id, user.name, user.lastname')
			->join('p.user', 'user')
			->join('p.groupCustomers', 'grp')
			->orderBy('user.lastname');
		$customers = $qb->getQuery()->getArrayResult();

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($customers));
	}

	/**
	 * @Path("/{id}")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Customer")
	 */
	public function updateCustomer(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->updateCustomer(
			$request->getParameter('id'),
			$request->getEntity(),
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(
			ArrayEntity::from([
				'result'     => (bool) $result,
				'customerId' => $result,
			]),
		);
	}

	/**
	 * @Path("/{id}/invoice-address/{address}")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 *      @RequestParameter(name="address", type="int", required=true),
	 * })
	 */
	public function setInvoiceAddressId(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->setCustomerInvoiceAddress(
			$request->getParameter('id'),
			$request->getParameter('address'),
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(
			ArrayEntity::from([
				'result' => $result,
			]),
		);
	}

	/**
	 * @Path("/{id}/delivery-address/{address}")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 *      @RequestParameter(name="address", type="int", required=true),
	 * })
	 */
	public function setDeliveryAddressId(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->setCustomerDeliveryAddress(
			$request->getParameter('id'),
			$request->getParameter('address'),
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(
			ArrayEntity::from([
				'result' => $result,
			]),
		);
	}

	/**
	 * @Path("/{id}")
	 * @Method("DELETE")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function deleteCustomer(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->deleteCustomer(
			$request->getParameter('id'),
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(
			ArrayEntity::from([
				'result' => $result,
			]),
		);
	}

	/**
	 * @Path("/search")
	 * @Method("POST")
	 */
	public function search(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$filter = $request->getJsonBody()['filter'];

		$ids = $this->apiCustomers->filterIds($filter);

		$customers = [];
		foreach ($ids as $id) {
			$customers[$id] = $this->apiCustomers->getCustomer($id);
		}

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($customers));
	}

	/**
	 * @Path("/{customer}/payments/{id}/disable")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function setDisabledPayment(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->setDisabledPayment(
			$request->getParameter('customer'),
			$request->getParameter('id'),
			(int) $request->getJsonBody()['value'],
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['result' => $result]));
	}

	/**
	 * @Path("/{customer}/speditions/{id}/disable")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function setDisabledSpedition(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->setDisabledSpedition(
			$request->getParameter('customer'),
			$request->getParameter('id'),
			(int) $request->getJsonBody()['value'],
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['result' => $result]));
	}

	/**
	 * @Path("/{customer}/customer-group")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 * })
	 */
	public function setCustomerGroup(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$value  = $request->getJsonBody()['value'];
		$result = $this->apiCustomers->setCustomerGroup(
			$request->getParameter('customer'),
			$value ? (int) $value : null,
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['result' => $result]));
	}

	/**
	 * TODO sifrovani?
	 * @Path("/{customer}/addresses")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 * })
	 */
	public function getAddresses(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(
				ArrayEntity::from(
					['addresses' => $this->apiCustomers->getAddresses($request->getParameter('customer'))],
				),
			);
	}

	/**
	 * @Path("/{customer}/addresses/{id}")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Address")
	 */
	public function editAddress(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->setAddress(
			$request->getParameter('customer'),
			$request->getEntity(),
			$request->getParameter('id'),
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['addressId' => $result]));
	}

	/**
	 * @Path("/{customer}/addresses")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Address")
	 */
	public function addAddress(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->setAddress(
			$request->getParameter('customer'),
			$request->getEntity(),
			null,
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['addressId' => $result]));
	}

	/**
	 * @Path("/{customer}/addresses/{id}")
	 * @Method("DELETE")
	 * @RequestParameters({
	 *      @RequestParameter(name="customer", type="int", required=true),
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function deleteAddress(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$result = $this->apiCustomers->deleteAddress(
			$request->getParameter('customer'),
			$request->getParameter('id'),
		);

		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['result' => $result]));
	}
}
