<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestBody;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use EshopOrders\ApiModule\Api\V1\Model\Orders;
use EshopOrders\Model\Event\OrderPayloadReturnEvent;
use Exception;

/**
 * @Path("/orders")
 */
class OrdersController extends BaseController
{
	public function __construct(
		protected EventDispatcher        $eventDispatcher,
		protected EntityManagerDecorator $em,
		protected Orders                 $orders,
	)
	{
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from($this->orders->getStatusesOverview()));
	}

	/**
	 * @Path("/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function getDetail(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		try {
			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['order' => $this->orders->getDetail($request->getParameter('id'))]));
		} catch (ErrorException $ex) {
			return $this->sendError($response, 'orderNotFound');
		}
	}

	/**
	 * @Path("/status/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from($this->orders->getIdByStatus($request->getParameter('status'))));
	}

	/**
	 * @Path("/status-not/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatusNot(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from($this->orders->getIdByStatusNot($request->getParameter('status'))));
	}

	/**
	 * @Path("/{id}/status")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\OrderStatus")
	 */
	public function setOrderStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(
				ArrayEntity::from([
					'result' => $this->orders->setOrderStatus(
						$request->getParameter('id'),
						$request->getEntity(),
					),
				]),
			);
	}

	/**
	 * @Path("/{id}/payment")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Payment")
	 */
	public function changePayment(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(
				ArrayEntity::from([
					'result' => $this->orders->changePayment(
							$request->getParameter('id'),
							$request->getEntity(),
						) !== null,
				]),
			);
	}

	/**
	 * @Path("/{id}/spedition")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Spedition")
	 */
	public function changeSpedition(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(
				ArrayEntity::from([
					'result' => $this->orders->changeSpedition(
							$request->getParameter('id'),
							$request->getEntity(),
						) !== null,
				]),
			);
	}

	/**
	 * @Path("/create")
	 * @Method("POST")
	 */
	public function create(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();

		try {
			$order  = $this->orders->createForCheckout($data);
			$result = ['success' => $order];

			$orderPayloadReturnEvent = new OrderPayloadReturnEvent($result, $order);
			$this->eventDispatcher->dispatch($orderPayloadReturnEvent, 'eshopOrders.api.orderPayloadReturnEvent');

			return $response->withStatus(ApiResponse::S200_OK)
				->withEntity(ArrayEntity::from($orderPayloadReturnEvent->data));
		} catch (Exception $e) {
			return $response->withStatus(ApiResponse::S200_OK)->withEntity(
				ArrayEntity::from([
					'error' => $e->getMessage(),
					'data'  => $data,
				]),
			);
		}
	}

}
