<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Model;

use Apitte\Core\Http\ApiResponse;
use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\Site;
use Core\Model\Event\EventDispatcher;
use Core\Model\Sites;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\ApiModule\Api\V1\BodyEntity\OrderStatus as BodyEntityOrderStatus;
use EshopOrders\ApiModule\Api\V1\BodyEntity\Payment;
use EshopOrders\ApiModule\Api\V1\BodyEntity\Spedition;
use EshopOrders\ApiModule\Api\V1\Exceptions\ErrorException;
use EshopOrders\ApiModule\Api\V1\Model\Dao;
use EshopOrders\ApiModule\Api\V1\Model\Dao\Address;
use EshopOrders\ApiModule\Api\V1\Model\Dao\OrderPayment;
use EshopOrders\ApiModule\Api\V1\Model\Dao\OrderSpedition;
use EshopOrders\ApiModule\Api\V1\Model\Helpers\CustomerHelper;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Orders as FrontOrders;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemSale;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Receipts;
use EshopOrders\Model\Speditions;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Utils\Helpers;
use EshopStock\Model\Repository\SupplyProducts;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\Floats;
use Nette\Utils\Validators;
use Tracy\Debugger;

class Orders
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected Countries              $countries,
		protected Statuses               $statuses,
		protected EventDispatcher        $eventDispatcher,
		protected FrontOrders            $frontOrders,
		protected CustomerHelper         $customerHelper,
		protected Statuses               $statusesService,
		protected Customers              $customersService,
		protected Sites                  $sitesService,
		protected Payments               $paymentsService,
		protected Speditions             $speditionsService,
		protected Receipts               $receipts,
		protected Invoices               $invoices,
	)
	{
	}

	public function getDetail(int $id): ?Dao\Order
	{
		$orderEntity = $this->frontOrders->getDetail($id);

		if (!$orderEntity || $orderEntity->isCorrectiveTaxDocument) {
			throw new ErrorException("Order '$id' not found", ApiResponse::S404_NOT_FOUND);
		}

		$order = new Dao\Order(
			$orderEntity->getId(),
			$orderEntity->getIdent(),
			$orderEntity->site->getIdent(),
			$orderEntity->lang,
			$orderEntity->getPrice(),
			$orderEntity->getPriceWithoutVat(),
		);

		$customer = $orderEntity->getCustomer();
		if ($customer) {
			$order->customerId = $customer->getId();
			$order->phone      = $customer->getPhone();

			$user             = $customer->getUser();
			$order->firstName = $user->name;
			$order->lastName  = $user->lastname;
			$order->email     = $user->email;

			if ($orderEntity->getCustomer()->getGroupCustomers()) {
				$order->customerGroupId = $orderEntity->getCustomer()->getGroupCustomers()->getId();
			}
		}

		$order->agreedTerms = (bool) $orderEntity->getAgreedTerms();
		$order->paid        = $orderEntity->paid;
		$order->message     = $orderEntity->getMessage();

		$order->deliveryAddress = $this->customerHelper->fillAddressByOrderAddress(
			$orderEntity->getAddressDelivery(),
			new Address(),
		);
		$order->invoiceAddress  = $this->customerHelper->fillAddressByOrderAddress(
			$orderEntity->getAddressInvoice(),
			new Address(),
		);

		foreach ($orderEntity->getOrderItems() as $item) {
			$orderItem = new Dao\OrderItem(
				$item->getId(),
				$item->getOrderItemText($order->lang)->getName(),
				$item->getQuantity(),
				$item->getPrice(),
				$item->getPriceWithoutVat(),
			);

			$orderItem->productId    = $item->getProductId();
			$orderItem->code1        = $item->getCode1();
			$orderItem->ean          = $item->getProduct() ? $item->getProduct()->ean : null;
			$orderItem->recyclingFee = $item->getRecyclingFeeWithoutVat();
			$orderItem->vatRate      = $item->getVatRate();
			$orderItem->moreData     = (array) $item->getMoreData();
			$orderItem->uploads      = $item->getUploadedFiles();

			foreach ($item->sales as $sale) {
				$orderItem->sales[] = new Dao\OrderItemSale(
					$sale->getId(), $sale->getValue(), $sale->getType(), $sale->getName(), $sale->getDescription()
				);
			}

			$order->items[] = $orderItem;
		}

		foreach ($orderEntity->getOrderDiscounts() as $item) {
			$orderDiscount = new Dao\OrderDiscount(
				$item->getId(),
				$item->getValue(),
				$item->getPrice(true),
				$item->getType()
			);

			$orderDiscount->code = $item->getCode();
			$orderDiscount->name = $item->getName();

			$order->discounts[] = $orderDiscount;
		}

		foreach ($orderEntity->getOrderStatuses() as $item) {
			$orderStatus = new Dao\OrderStatus(
				$item->getStatus()->getId(),
				$item->getCreated()
			);

			$orderStatus->parameter = $item->getParameter();
			$orderStatus->message   = $item->getMessage();

			$order->statuses[$item->getStatus()->getId()] = $orderStatus;
		}

		foreach ($orderEntity->getOrderFlags() as $item) {
			$orderFlag = new Dao\OrderFlag(
				$item->getType(),
				(bool) $item->getState(),
			);

			$order->flags[$item->getType()] = $orderFlag;
		}

		foreach ($orderEntity->getGifts() as $item) {
			$orderGift = new Dao\OrderGift(
				$item->getName()
			);

			$orderGift->product = $item->getProductId();
			$orderGift->code1   = $item->code1;
			$orderGift->ean     = $item->ean;

			$order->gifts[] = $orderGift;
		}

		if ($orderEntity->getPayment()) {
			$payment        = new OrderPayment();
			$payment->id    = $orderEntity->getPayment()->getPaymentId();
			$payment->ident = $orderEntity->getPayment()->getIdent();
			$payment->name  = $orderEntity->getPayment()->getName();
			$payment->price = $orderEntity->getPayment()->getPrice();

			$order->payment = $payment;
		}

		if ($orderEntity->getSpedition()) {
			$spedition        = new OrderSpedition();
			$spedition->id    = $orderEntity->getSpedition()->getSpeditionId();
			$spedition->ident = $orderEntity->getSpedition()->getIdent();
			$spedition->name  = $orderEntity->getSpedition()->getName();
			$spedition->price = $orderEntity->getSpedition()->getPrice();

			$order->spedition = $spedition;
		}

		return $order;
	}

	public function getIdByStatus(string $status): array
	{
		return array_map(
			static fn($row) => $row['id'],
			$this->em->getConnection()->fetchAllAssociative(
				"SELECT s.order_id as id, s.status_id as status FROM eshop_orders__order_status s
			WHERE s.created = (SELECT MAX(s2.created) FROM eshop_orders__order_status s2 WHERE s.order_id = s2.order_id)
			AND s.status_id = '{$status}'
			ORDER BY s.order_id DESC",
			),
		);
	}

	public function getIdByStatusNot(string $status): array
	{
		return array_map(
			static fn($row) => $row['id'],
			$this->em->getConnection()->fetchAllAssociative(
				"SELECT s.order_id as id, s.status_id as status FROM eshop_orders__order_status s
			WHERE s.created = (SELECT MAX(s2.created) FROM eshop_orders__order_status s2 WHERE s.order_id = s2.order_id)
			AND s.status_id NOT IN ('" . implode("', '", explode(',', $status)) . "')
			ORDER BY s.order_id DESC",
			),
		);
	}

	public function setOrderStatus(int $orderId, BodyEntityOrderStatus $orderStatus): bool
	{
		$statusId = null;
		/** @var Status|null $status */
		$status = $this->em->getRepository(Status::class)->find($orderStatus->status);
		if (!$status) {
			Debugger::log(
				"Status not found - order '{$orderId} status '{$orderStatus->status}",
				'api-eshoporders-order',
			);

			return false;
		}

		/** @var Order|null $order */
		$order = $this->em->getRepository(Order::class)->find($orderId);
		if (!$order) {
			Debugger::log(
				"Order not found - order '{$orderId} status '{$orderStatus->status}",
				'api-eshoporders-order',
			);

			return false;
		}

		try {
			$newStatus = new OrderStatus(
				$order,
				$status,
			);

			if ($orderStatus->message) {
				$newStatus->setMessage($orderStatus->message);
			}

			$this->em->persist($newStatus);
			$this->em->flush();
		} catch (Exception) {
			Debugger::log("Change status failed - order '{$orderId} status '{$statusId}", 'api-eshoporders-order');

			return false;
		}

		if ($orderStatus->sendToCustomer) {
			$this->statuses->sendOrderStatusEmail($order, $newStatus);
		}

		return true;
	}

	public function getStatusesOverview(): array
	{
		$result = [];

		$usedIds = [];
		$qb      = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('s')
			->select('IDENTITY(s.order) as order, IDENTITY(s.status) as status')
			->where('s.deleted IS NULL')
			->orderBy('s.created', 'DESC');

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			if (isset($usedIds[$row['order']])) {
				continue;
			}

			$result[$row['status']][] = $row['order'];
			$usedIds[$row['order']]   = $row['order'];
		}
		$usedIds = null;

		foreach ($result as $status => $ids) {
			$result[$status] = array_reverse($ids);
		}

		return $result;
	}

	public function convertJSONProductsToOrderItems(array $JSONproducts, Order $order): array
	{
		$result       = [];
		$isCorrective = false;

		$removedOrderItemIds = array_diff(
			array_map(static fn(OrderItem $oi) => $oi->getId(), $order->getOrderItems()->toArray()),
			array_filter(array_map(static fn(array $data) => $data['orderItemId'], $JSONproducts), static fn($val) => !Validators::isNone($val))
		);

		foreach ($removedOrderItemIds as $oiId) {
			$this->em->remove($this->em->getReference(OrderItem::class, $oiId));
		}

		foreach ($JSONproducts as $item) {
			$product = $this->em->getReference(Product::class, $item['id']);
			$lang    = $this->translator->getLocale();
			$name    = $item['name'];

			if (EshopCatalogConfig::load('addManufacturerBeforeProductName', false) && $product->getManufacturer()) {
				$name = trim($product->getManufacturer()->name . ' ' . $name);
			}

			$price    = (float) $item['price'];
			$quantity = (int) $item['count'];

			if (!Validators::isNone($item['id'])) {
				$orderItem = $this->em->getRepository(OrderItem::class)->find((int) $item['orderItemId']);
			}
			$orderItem = $orderItem ?? new OrderItem($product, $order);
			$orderItem->setQuantity(abs($quantity));
			if ($orderItem->getId() === null) {
				$orderItem->addOrderItemText($lang);
			}
			$orderItem->getOrderItemText($lang)->setName($name);
			if ($quantity < 0 || $isCorrective) { // pokud ma alespon jeden produkt zapornou kvantitu, pak se objednavka stava oprav. dan. dokladem
				$price        = -$price;
				$isCorrective = true;
			}

			$orderItem->setPrice($price);
			$orderItem->recyclingFee = $product->recyclingFee;
			$orderItem->setVatRate($item['vatRate']);
			$orderItem->setCode1($item['code1']);

			$this->addOrderItemsSales($item, $orderItem);

			if (!Validators::isNone($item['sale']['all']['id']) && Floats::areEqual((float) $item['sale']['all']['staticSale'], 0) && Floats::areEqual((float) $item['sale']['all']['percentSale'], 0)) {
				$ois = $this->em->getRepository(OrderItemSale::class)->find($item['sale']['all']['id']);
				if ($ois) {
					$this->em->remove($ois);
				}
			}

			if (!Validators::isNone($item['sale']['one']['id']) && Floats::areEqual((float) $item['sale']['one']['staticSale'], 0) && Floats::areEqual((float) $item['sale']['one']['percentSale'], 0)) {
				$ois = $this->em->getRepository(OrderItemSale::class)->find($item['sale']['one']['id']);
				if ($ois) {
					$this->em->remove($ois);
				}
			}

			$this->em->persist($orderItem);
			$result[] = $orderItem;
		}

		return $result;
	}

	public function addGlobalSales(array $globalSales, Order $order, $totalPrice): void
	{
		foreach ($globalSales as $sale) {
			$saleEntity = null;

			if (!Validators::isNone($sale['id'])) {
				$saleEntity = $this->em->getReference(OrderDiscount::class, (int) $sale['id']);
			}

			$saleEntity = $saleEntity ?? new OrderDiscount('CHECKOUTSALE', $order);
			$saleEntity->setName($sale['name']);

			if (!empty($sale['saleCode'])) {
				$saleEntity->setCode($sale['saleCode']);
			}

			// procentualni sleva
			if ($sale['percentSale'] > 0) {
				$saleEntity->setValue((float) $sale['percentSale']);
				$saleEntity->setType(OrderDiscount::TYPE_PERCENT);
				$price = (100 / (100 - (float) $sale['percentSale'])) * $totalPrice; // cena pred uplatneni slevy
				$saleEntity->setPrice($price - $totalPrice);
			}

			// pevna sleva
			if ($sale['staticSale'] > 0) {
				$saleEntity->setValue((float) $sale['staticSale']);
				$saleEntity->setType(OrderDiscount::TYPE_FIX);
				$saleEntity->setPrice((float) $sale['staticSale']);
			}

			if ($saleEntity->getId() === null) {
				$order->addOrderDiscount($saleEntity);
			}
			$this->em->persist($saleEntity);
		}
	}

	/**
	 * @param $item
	 * @param $orderItem
	 */
	public function addOrderItemsSales($item, $orderItem): void
	{

		// Pevna sleva na vsechny kusy
		if (($staticSale = $item['sale']['all']['staticSale']) > 0) {
			$sale = null;
			if (!Validators::isNone($item['sale']['all']['id'])) {
				$sale = $this->em->getReference(OrderItemSale::class, (int) $item['sale']['all']['id']);
			}
			$sale = $sale ?? new OrderItemSale($staticSale, $orderItem, OrderItemSale::TYPE_FIX_ALL);
			$sale->setValue((float) $staticSale);
			$sale->setType(OrderItemSale::TYPE_FIX_ALL);
			$this->em->persist($sale);
		}

		// Procentuelni sleva na vsechny kusy
		if (($percentSale = $item['sale']['all']['percentSale']) > 0) {
			$sale = null;
			if (!Validators::isNone($item['sale']['all']['id'])) {
				$sale = $this->em->getReference(OrderItemSale::class, (int) $item['sale']['all']['id']);
			}
			$sale = $sale ?? new OrderItemSale($percentSale, $orderItem, OrderItemSale::TYPE_PERCENT_ALL);
			$sale->setValue((float) $percentSale);
			$sale->setType(OrderItemSale::TYPE_PERCENT_ALL);
			$this->em->persist($sale);
		}

		// Pevna sleva na kus
		if (($staticSale = $item['sale']['one']['staticSale']) > 0) {
			$sale = null;
			if (!Validators::isNone($item['sale']['one']['id'])) {
				$sale = $this->em->getReference(OrderItemSale::class, (int) $item['sale']['one']['id']);
			}
			$sale = $sale ?? new OrderItemSale($staticSale, $orderItem, OrderItemSale::TYPE_FIX_ONE);
			$sale->setValue((float) $staticSale);
			$sale->setType(OrderItemSale::TYPE_FIX_ONE);
			$this->em->persist($sale);
		}

		// Procentuelni sleva na kus
		if (($percentSale = $item['sale']['one']['percentSale']) > 0) {
			$sale = null;
			if (!Validators::isNone($item['sale']['one']['id'])) {
				$sale = $this->em->getReference(OrderItemSale::class, (int) $item['sale']['one']['id']);
			}
			$sale = $sale ?? new OrderItemSale($percentSale, $orderItem, OrderItemSale::TYPE_PERCENT_ONE);
			$sale->setValue((float) $percentSale);
			$sale->setType(OrderItemSale::TYPE_PERCENT_ONE);
			$this->em->persist($sale);
		}
	}

	public function createAddress(OrderAddress $address, array $addressData): OrderAddress
	{

		$address->setFirstName($addressData['firstName'] ?? '');
		$address->setLastName($addressData['lastName'] ?? '');
		$address->setEmail($addressData['email'] ?? '');
		$address->setPhone($addressData['phone'] ?? '');
		$address->setStreet($addressData['street'] ?? '');
		$address->setCity($addressData['city'] ?? '');
		$address->setPostal($addressData['postal'] ?? '');
		if (!empty($addressData['country'])) {
			$address->setCountry($this->countries->getReference($addressData['country']));
		}
		$address->setCompany($addressData['company'] ?? '');
		$address->setIdNumber($addressData['idNumber'] ?? '');
		$address->setVatNumber($addressData['vatNumber'] ?? '');
		$this->em->persist($address);
		bdump($address);

		return $address;
	}

	public function changePayment(int $orderId, Payment $data): ?array
	{
		/** @var ?Order $order */
		$order = $this->em->getRepository(Order::class)->find($orderId);

		if (!$order) {
			return null;
		}

		try {
			if ($data->paymentId) {
				/** @var ?\EshopOrders\Model\Entities\Payment $payment */
				$payment = $this->em->getRepository(\EshopOrders\Model\Entities\Payment::class)->find($data->paymentId);

				if ($payment) {
					$orderPayment = $order->getPayment();

					if (!$orderPayment) {
						$orderPayment = new \EshopOrders\Model\Entities\OrderPayment(
							$payment,
							$order,
						);
						$orderPayment->setVatRate(21);
						$orderPayment->setPrice((float) $payment->getPrice());

						$order->setPayment($orderPayment);
					} else {
						$orderPayment->setPayment($payment);
					}

					$orderPayment->setName($payment->getName());

					$this->em->persist($orderPayment);
				} else {
					return null;
				}
			}

			if ($data->price !== null) {
				$orderPayment = $order->getPayment();
				if ($orderPayment) {
					$orderPayment->setPrice($data->price);
					$this->em->persist($orderPayment);
				}
			}

			$this->em->flush();

			return [];
		} catch (Exception $e) {
			Debugger::log($e, Debugger::EXCEPTION);
		}

		return null;
	}

	public function changeSpedition(int $orderId, Spedition $data): ?array
	{
		/** @var ?Order $order */
		$order = $this->em->getRepository(Order::class)->find($orderId);

		if (!$order) {
			return null;
		}

		try {
			if ($data->speditionId) {
				/** @var ?\EshopOrders\Model\Entities\Spedition $spedition */
				$spedition = $this->em->getRepository(\EshopOrders\Model\Entities\Spedition::class)
					->find($data->speditionId);

				if ($spedition) {
					$orderSpedition = $order->getSpedition();

					if (!$orderSpedition) {
						$orderSpedition = new \EshopOrders\Model\Entities\OrderSpedition(
							$spedition,
							$order,
						);
						$orderSpedition->setVatRate(21);
						$orderSpedition->setPrice((float) $spedition->getPrice());

						$order->setSpedition($orderSpedition);
					} else {
						$orderSpedition->setSpedition($spedition);
					}

					$orderSpedition->setName($spedition->getName());

					$this->em->persist($orderSpedition);
				} else {
					return null;
				}
			}

			if ($data->price !== null) {
				$orderSpedition = $order->getSpedition();
				if ($orderSpedition) {
					$orderSpedition->setPrice($data->price);
					$this->em->persist($orderSpedition);
				}
			}

			$this->em->flush();

			return [];
		} catch (Exception $e) {
			Debugger::log($e, Debugger::EXCEPTION);
		}

		return null;
	}

	public function createForCheckout(array $data): Order
	{
		try {

			$this->em->getConnection()->beginTransaction();

			$order = null;
			if (!Validators::isNone($data['orderId'])) {
				$order = $this->em->getRepository(Order::class)->find((int) $data['orderId']);
			}

			$order = $order ?? new Order($this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent()));

			$event                 = new OrderEvent($order);
			$event->orderPayment   = $order->getPayment();
			$event->orderSpedition = $order->getSpedition();
			$event->addrDelivery   = $order->getAddressDelivery();
			$event->addrInvoice    = $order->getAddressInvoice();
			$event->orderItems     = $order->getOrderItems()->toArray();
			$event->formData       = $data;
			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.beforeOrderChange');

			$order->setMessage($data['globalNote'] ?? '');
			$order->setAgreedTerms(true);

			AppState::setState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.checkout');
			$order->isPaid = 1;
			$order->paid   = new DateTime;
			$orderItems    = $this->convertJSONProductsToOrderItems($data['checkoutProducts'], $order);
			$isCorrective  = Helpers::isCorrectiveByOrderItems($orderItems);
			$order->setOrderItems($orderItems);
			$order->isCorrectiveTaxDocument = (int) $isCorrective;

			$payment      = $this->paymentsService->getReferenceByIdent($data['payment']);
			$orderPayment = $order->getPayment() ?? new \EshopOrders\Model\Entities\OrderPayment($payment, $order);
			$orderPayment->setName($payment->getName());
			$orderPayment->setPrice((float) $payment->getPrice());
			$orderPayment->setPayment($payment);

			$spedition      = $this->speditionsService->getReferenceByIdent($data['spedition']);
			$orderSpedition = $order->getSpedition() ?? new \EshopOrders\Model\Entities\OrderSpedition($spedition, $order);
			$orderSpedition->setName($spedition->getName());
			$orderSpedition->setPrice((float) $spedition->getPrice());
			$orderSpedition->setSpedition($spedition);

			$orderStatuses      = [];
			$hasStatusCreated   = false;
			$hasStatusSpedition = false;
			$hasStatusFinished  = false;
			if ($order->getId() !== null) {
				foreach ($order->getOrderStatuses()->toArray() as $orderStatus) {
					if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CREATED) {
						$hasStatusCreated = true;
					} else if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_SPEDITION) {
						$hasStatusSpedition = true;
					} else if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_FINISHED) {
						$hasStatusFinished = true;
					}
				}
			}

			$orderActionCreated = null;
			if (!$hasStatusCreated) {
				$statusCreated      = $this->statusesService->get(OrderStatus::STATUS_CREATED);
				$orderActionCreated = new OrderStatus($order, $statusCreated);
				$orderStatuses[]    = $orderActionCreated;
			}

			$orderActionSpedition = null;
			if (!$hasStatusSpedition) {
				$statusSpedition      = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
				$orderActionSpedition = new OrderStatus($order, $statusSpedition);
			}

			$orderActionFinished = null;
			if (!$hasStatusFinished) {
				$statusFinished      = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
				$orderActionFinished = new OrderStatus($order, $statusFinished);
				$orderStatuses[]     = $orderActionFinished;
			}

			$order->setPayment($orderPayment);
			$order->setSpedition($orderSpedition);

			$addressInvoice  = $this->createAddress(
				new OrderAddress(OrderAddress::ADDRESS_INVOICE),
				$data['customer']['addressInvoice'],
			);
			$addressDelivery = $this->createAddress(
				new OrderAddress(OrderAddress::ADDRESS_DELIVERY),
				$data['customer']['addressDelivery'],
			);
			$order->setAddressInvoice($addressInvoice);
			$order->setAddressDelivery($addressDelivery);

			if ($data['customer']['customerId'] && $customer = $this->customersService->get(
					(int) $data['customer']['customerId'],
				)) {
				$order->setCustomer($customer);
				$this->em->persist($customer);
			}

			$removedOrderDiscountIds = array_diff(
				array_map(static fn(OrderDiscount $oi) => $oi->getId(), $order->getOrderDiscounts()->toArray()),
				array_filter(array_map(static fn(array $data) => $data['id'], $data['globalSales']), static fn($val) => !Validators::isNone($val))
			);
			foreach ($removedOrderDiscountIds as $discountId) {
				$orderDiscount = $this->em->getReference(OrderDiscount::class, (int) $discountId);
				if ($orderDiscount) {
					$this->em->remove($orderDiscount);
				}
			}
			$this->addGlobalSales($data['globalSales'], $order, $data['totalPrice']);

			/*if ($data['customer']['customerCreated'] && ($data['customer']['delivery']['email'] || $data['customer']['invoice']['email'])) {
				// Id vkladam jenom formalne - nemel by existovat
				$customer = $this->customersService->getOrCreateCustomer($data['customer']['customerId']);
				$customer->setAddressDelivery($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$customer->setAddressInvoice($this->createAddress(new CustomerAddress($customer), $data['customer']['delivery']));
				$this->em->persist($customer);
			}*/

			$event                 = new OrderEvent($order);
			$event->orderPayment   = $orderPayment;
			$event->orderSpedition = $orderSpedition;
			$event->addrDelivery   = $addressDelivery;
			$event->addrInvoice    = $addressInvoice;
			$event->orderItems     = $orderItems;
			$event->formData       = $data;

			$eshopStockSupplyProduct = class_exists('EshopStock\DI\EshopStockExtension') ? SupplyProducts::class : null;

			if ($eshopStockSupplyProduct !== null) {
				$eshopStockSupplyProduct::$assignOrderItemFlush = false;
			}

			$this->eventDispatcher->dispatch($event, 'eshopCheckout.orderBeforeSave');

			if (EshopOrdersConfig::load('receipt.enableUseNumericalSeries', false) && in_array(
					$orderPayment->getIdent(),
					['storeCard',
						'storeCash'],
				)) {
				$order->receiptIdent = $this->receipts->generateIdent($order);
			}

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);
			if ($orderActionCreated) {
				$this->em->persist($orderActionCreated);
			}
			if ($orderActionSpedition) {
				$this->em->persist($orderActionSpedition);
			}
			if ($orderActionFinished) {
				$this->em->persist($orderActionFinished);
			}

			$this->em->flush();

			if ($isCorrective) {
				$invoice = $this->invoices->getOneByOrder($order, true, InvoiceConfig::TYPE_CORRECTIVE);
				$order->setInvoice($invoice);
				$this->em->flush();
				$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.correctiveTaxDocumentCreated');
			}

			$this->em->getConnection()->commit();

			$this->eventDispatcher->dispatch($event, 'eshopOrders.admin.orderOnSuccess');

			return $order;
		} catch (Exception $exception) {
			Debugger::log($exception);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}

			throw new $exception;
		}
	}

}
