<?php declare(strict_types = 1);

namespace EshopOrders\Components;

use Core\Model\Countries;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\FormHelper;
use Core\Model\Module;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\EshopOrdersConfig;
use Exception;
use Nette\Utils\ArrayHash;
use Users\Model\Security\User;

class AddressForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $addressId = null;

	/** @var int|null @persistent */
	public ?int $customerId = null;

	/** @var null|array */
	protected ?array $idNumberData = null;

	protected ?CustomerAddress $address = null;

	public function __construct(protected Countries $countries, protected User $user, protected Customers $customers)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$countries = $this->countries->getAllNameColumn();

		$form->addText('addressName', 'eshopOrders.customerAddress.addressName')
			->setMaxLength(255);
		$form->addText('firstName', 'eshopOrders.default.firstName')
			->setMaxLength(255);
		$form->addText('lastName', 'eshopOrders.default.lastName')
			->setMaxLength(255);
		$form->addText('email', 'eshopOrders.default.email')
			->setMaxLength(255);
		$form->addText('phone', 'eshopOrders.default.phone')
			->setMaxLength(255);
		$form->addText('street', 'eshopOrders.default.street')
			->setMaxLength(255);
		$form->addText('city', 'eshopOrders.default.city')
			->setMaxLength(255);
		$form->addText('postal', 'eshopOrders.default.postal')
			->setMaxLength(255);
		$form->addSelect('country', 'eshopOrders.default.country', $countries);
		$form->addText('company', 'eshopOrders.default.company')
			->setMaxLength(255);
		$form->addText('idNumber', 'eshopOrders.default.idNumber')
			->setMaxLength(255);
		$form->addText('vatNumber', 'eshopOrders.default.vatNumber')
			->setMaxLength(255);
		if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
			$form->addText('idVatNumber', 'eshopOrders.customerAddress.idVatNumber')
				->setMaxLength(255);
		}

		if (EshopOrdersConfig::load('customer.disableModifyIco') && Module::isFront()) {
			$form['idNumber']->setDisabled();
			$form['vatNumber']->setDisabled();

			if ($this->user->isLoggedIn()) {
				$icoData = [
					'idNumber'  => null,
					'vatNumber' => null,
				];

				/** @var CustomerAddress[] $addressesList */
				$addressesList = $this->em->getRepository(CustomerAddress::class)->createQueryBuilder('ca')
					->innerJoin('ca.customers', 'c', Join::WITH, 'c.user = :user')
					->setParameter('user', $this->user->getId())
					->getQuery()->getArrayResult();

				foreach ($addressesList as $addr) {
					foreach ($icoData as $colKey => $colVal) {
						if (!$colVal && $addr[$colKey]) {
							$icoData[$colKey] = $addr[$colKey];
							$form[$colKey]->setValue($addr[$colKey]);
						}
					}
				}
			}
		}

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$customer = $this->em->getRepository(Customer::class)->find($this->customerId);

			if ($this->addressId) {
				$address      = $this->em->getRepository(CustomerAddress::class)->find($this->addressId);
				$flashMessage = 'eshopOrders.customerAddress.edited';
			} else {
				/** @var Customer $customer */
				$address      = new CustomerAddress($customer);
				$flashMessage = 'eshopOrders.customerAddress.created';
			}

			$address->country = $values->country ? $this->countries->getReference($values->country) : null;

			unset($values->country);
			if (EshopOrdersConfig::load('customer.disableModifyIco') && Module::isFront()) {
				unset($values->idNumber);
				unset($values->vatNumber);

				foreach ($this->getIdNumberData() as $col => $v) {
					$values->{$col} = $v;
				}
			}

			FormHelper::fillEntityByValues($address, $values);

			$this->em->persist($address);
			$this->em->flush();

			$customer->addressesAvailable->removeElement($address);
			$customer->addressesAvailable->set($address->getId(), $address);

			$form->addCustomData('addressId', $address->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setAddress(int $addressId): void
	{
		$this->addressId = $addressId;
		/** @var CustomerAddress $address */
		$address = $this->em->getRepository(CustomerAddress::class)->find($addressId);

		if (!$addressId) {
			$this->error();
		}

		$skip = [];
		if (Module::isFront()) {
			$customer = $this->customers->getByUser($this->user->getIdentity());
			if (!$customer || !Arrays::contains($address->customers->getKeys(), $customer->getId())) {
				$this->error();
			}

			if (EshopOrdersConfig::load('customer.disableModifyIco')) {
				$skip = ['idNumber', 'vatNumber'];
			}
		}

		FormHelper::fillContainerByEntity($this['form'], $address, $skip);
	}

	public function getIdNumberData(): array
	{
		if ($this->idNumberData === null) {
			$this->idNumberData = [];

			if ($this->customerId) {
				foreach ($this->em->getRepository(CustomerAddress::class)->createQueryBuilder('ca')
					         ->select('ca.idNumber, ca.vatNumber, ca.company')
					         ->innerJoin('ca.customers', 'c', Join::WITH, 'c.id = :id')
					         ->setParameter('id', $this->customerId)
					         ->getQuery()->getArrayResult() as $row) {
					foreach (['idNumber', 'vatNumber'] as $col) {
						if (!isset($this->idNumberData[$col]) && $row[$col]) {
							$this->idNumberData[$col] = $row[$col];
						}
					}
				}
			}
		}

		return $this->idNumberData;
	}
}
