<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\UI\BaseControl;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Application\UI\Multiplier;
use Nette\DI\Container;

class CartDetail extends BaseControl
{
	public function __construct(
		protected CartFacade             $cartFacade,
		protected ICartDetailItemFactory $cartDetailItemFactory,
		protected Container              $container,
	)
	{
		// add discount form control to this component
		if ($container->hasService('eshopSales.front.discountsGridFactory')) {
			$discountsGridControl = $container->getService('eshopSales.front.discountsGridFactory');
			if ($discountsGridControl !== null) {
				$this->addComponent($discountsGridControl->create(), 'discountsGrid');
			}
		}

		if ($container->hasService('eshopSales.front.discountFormFactory')) {
			$discountFormControl = $container->getService('eshopSales.front.discountFormFactory');
			if ($discountFormControl !== null) {
				$this->addComponent($discountFormControl->create(), 'discountForm');
			}
		}
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', function() {
			$cart = $this->cartFacade->getCart();

			if ($cart->getItemsCount())
				$this->redrawControl('wrap');
			else
				$this->getPresenter()->redrawControl('cart');
		});

		$this->eventDispatcher->addListener('eshopSales.discountCodesChanged', function() {
			$this->getParent()->redrawControl('cart');
		});
	}

	public function handleRemove($itemId): void
	{
		$cartItems = $this->cartFacade->getCartsService()->getCurrentCart()->getCartItems();
		$event     = new UpdatedCartItemEvent(
			(int) $itemId,
			(int) 0,
			(int) $cartItems[$itemId]->quantity,
			[],
			(int) $cartItems[$itemId] ? $cartItems[$itemId]->getProductId() : null,
		);

		$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
	}

	public function handleChooseOrderGift(int $giftId)
	{
		$this->cartFacade->getCartsService()->setOrderGift($giftId);
		$this->redrawControl('giftsList');
	}

	public function handleRemoveOrderGift()
	{
		$this->cartFacade->getCartsService()->setOrderGift();
		$this->redrawControl('giftsList');
	}

	public function render(): void
	{
		$cart                      = $this->cartFacade->getCart();
		$this->template->cart      = $cart;
		$this->template->orderGift = $cart->getOrderGift();

		$style = EshopOrdersConfig::load('cartDetailFreeSped.style', 1);
		if ($style === 1) {
			$freeSped                               = $this->cartFacade->countFreeSpedition();
			$this->template->freeSpeditionPrice     = $freeSped['minPrice'];
			$this->template->freeSpeditionRemaining = $freeSped['remaining'];
		} else if ($style === 2) {
			$this->template->freeSped = $this->cartFacade->countFreeSpeditionMilestones();
		}

		$this->eventDispatcher->dispatch(
			new ComponentTemplateEvent($this->template, $this),
			'eshopOrders.cartDetail.render',
		);

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentCartDetailItem(): Multiplier
	{
		return new Multiplier(function($item) {
			$control = $this->cartDetailItemFactory->create();
			$control->setItem((int) $item);

			return $control;
		});
	}

}

