<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\Helpers\Arrays;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use Exception;
use Nette\Utils\ArrayHash;

class CartDetailItem extends BaseControl
{
	protected ?int $itemId = null;

	public function __construct(protected Carts $cartsService, protected Sites $sites)
	{
	}

	public function render(): void
	{
		$this->template->item = $this->getItem();
		$this->template->render($this->getTemplateFile(null, 'CartDetailItem'));
	}

	public function handleRemove(): void
	{
		$cartItems = $this->cartsService->getCurrentCart()->getCartItems();
		$event     = new UpdatedCartItemEvent(
			(int) $this->getItem()->getId(),
			(int) 0,
			(int) $cartItems[$this->getItem()->getId()]->quantity,
			[],
			(int) $this->getItem()->getProductId(),
		);

		$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
	}

	protected function createComponentForm(): ?BaseForm
	{
		$form          = $this->createForm();
		$item          = $this->getItem();
		$minimumAmount = $item->getProduct() ? $item->getProduct()->minimumAmount : 1;
		$maximumAmount = $item->getProduct() ? $item->getProduct()->getMaximumAmount() : null;
		$site          = $this->sites->getCurrentSite();

		$form->addText('quantity', null)
			->addRule($form::FILLED)
			->setHtmlAttribute('data-add-to-cart-quantity-input', 'instant')
			->setHtmlAttribute('data-item-id', $item->getId())
			->setDefaultValue($item->quantity);
		$form['quantity']
			->setHtmlAttribute('data-minimum-amount', $minimumAmount)
			->addRule(
				BaseForm::MIN,
				'eshopCatalogFront.cart.minimumQuantity',
				$minimumAmount > 1 ? $minimumAmount : 0,
			);
		$form->addHidden('itemId', (string) $item->getId());

		$product                       = $item->product;
		$cartProductMaxQuantityByEshop = Config::load('cartProductMaxQuantityByEshop', []) ?? [];

		if (
			(($product
					&& Config::load('pseudoWarehouse')
					&& $product->unlimitedQuantity === 0
					&& $product->getQuantity() > 0
				)
				&& (!$maximumAmount || $maximumAmount > $product->getQuantity() + $product->getQuantityExternal())
				&& (!Config::load('allowWarehouseOverdraft') || !$product->getAvailability()->warehouseOverdraft))
			|| (
				Arrays::contains($cartProductMaxQuantityByEshop, $site->getIdent())
				&& (!$product->availabilityAfterSoldOut || !$product->availability->canAddToCart())
			)
		) {
			$maximumAmount = $product->getQuantity() + $product->getQuantityExternal();
		}

		if ($maximumAmount) {
			$form['quantity']->addRule(BaseForm::MAX, 'eshopCatalogFront.cart.maximumQuantity', $maximumAmount);
			$form['quantity']->setHtmlAttribute('data-max', $maximumAmount);
		}

		if ($item->getData('note')) {
			$form->addTextArea('note', $item->getData('noteTitle'))
				->setDefaultValue($item->getData('note'));
		}

		$childsContainer = $form->addContainer('childs');
		foreach ($item->getChilds() as $child) {
			$childContainer = $childsContainer->addContainer('child' . $child->getId());

			if ($child->getData('note'))
				$childContainer->addTextArea('note', $child->getData('noteTitle'))
					->setDefaultValue($child->getData('note'));
		}

		$form->onSuccess[] = $this->formOnSuccess(...);

		return $form;
	}

	public function formOnSuccess($form, ArrayHash $values): bool
	{
		try {
			$cartItems = $this->cartsService->getCurrentCart()->getCartItems();
			$event     = new UpdatedCartItemEvent(
				(int) $values->itemId,
				(int) $values->quantity,
				(int) $cartItems[$values->itemId]->quantity,
				[],
				(int) $cartItems[$values->itemId]->productId,
			);

			if ($values->note) {
				$event->moreData['note'] = $values->note;
			}

			$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');

			foreach ($values->childs ?? [] as $childId => $childVals) {
				if (empty((array) $childVals))
					continue;

				$childId = substr((string) $childId, 5);
				$child   = $cartItems[$values->itemId]->getChilds()[$childId];

				$event = new UpdatedCartItemEvent(
					(int) $child->getId(),
					(int) $child->quantity,
					(int) $child->quantity,
					[],
					(int) $child->productId,
				);

				if ($childVals->note)
					$event->moreData['note'] = $childVals->note;

				$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
			}
		} catch (Exception) {
			$this->getPresenter()->flashMessageWarning('eshopOrders.cart.countUpdateErr');
			$this->getPresenter()->redrawControl('flashes');

			return false;
		}

		return true;
	}

	protected function getItem(): ?CartItem
	{
		return $this->cartsService->getCartItem($this->itemId);
	}

	public function setItem(int $itemId): self
	{
		$this->itemId = $itemId;

		return $this;
	}
}

