<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Components\AddressForm;
use EshopOrders\Components\IAddressFormFactory;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\EshopOrdersConfig;

class AddressesList extends BaseControl
{
	protected ?Customer $customer = null;

	/** @var callable[] */
	public array $onAddressSave = [];

	public function __construct(
		protected int                 $userId,
		protected Customers           $customers,
		protected IAddressFormFactory $addressFormFactory,
	)
	{
	}

	public function render(): void
	{
		$customer = $this->getCustomer();
		if (!$customer) {
			return;
		}

		$list = [];
		foreach ($customer->addressesAvailable as $item) {
			if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')
				&& $customer->getAddressInvoice()
				&& $item->getId() === $customer->getAddressInvoice()->getId()) {
				continue;
			}

			$list[$item->getId()] = $item;
		}

		$this->template->addresses = $list;
		$this->template->render($this->getTemplateFile());
	}

	protected function getCustomer(): ?Customer
	{
		if ($this->customer === null)
			$this->customer = $this->customers->getByUserId($this->userId);

		return $this->customer;
	}

	/** ================================================================================================================
	 * ===========  Handle
	 */

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle      = $this->t('eshopOrdersFront.customer.addressEdit');
		$this['addressForm']->addressId  = $id;
		$this['addressForm']->customerId = $this->getCustomer()->getId();
		$this->template->showModal       = true;
		$this->redrawControl('modal');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle      = $this->t('eshopOrdersFront.customer.addNewAddress');
		$this['addressForm']->customerId = $this->getCustomer()->getId();
		$this->template->showModal       = true;
		$this->redrawControl('modal');
	}

	public function handleRemove(int $id): void
	{
		$this->em->remove($this->em->getReference(CustomerAddress::class, $id));
		$this->em->flush();
		$this->getPresenter()->flashMessageSuccess('eshopOrdersFront.customer.addressRemoved');
		$this->getPresenter()->redrawControl('fleshed');
		$this->redrawControl('addresses');
	}

	/** ================================================================================================================
	 * ===========  Components
	 */

	protected function createComponentAddressForm(): AddressForm
	{
		$control = $this->addressFormFactory->create();

		if ($this->getParameter('id')) {
			$control->setAddress((int) $this->getParameter('id'));
		}

		$control['form']['saveControl']->removeComponent($control['form']['saveControl']['cancel']);
		$control['form']['saveControl']->removeComponent($control['form']['saveControl']['save']);
		$control['form']->setShowLangSwitcher(false);

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->template->showModal                = false;
			$this->redrawControl('modal');
			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('addresses');
			$this->onAddressSave($this);
		};

		return $control;
	}
}
