<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Order;

use Core\Model\Application\AppState;
use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\Entities\Site;
use Core\Model\Event\Event;
use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\PhoneInput;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Helpers\Psc;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use EshopOrders\Model\PaymentSpeditions;
use EshopOrders\Model\Statuses;
use Exception;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Security\User;
use Nette\Utils\ArrayHash;
use Nette\Utils\Floats;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Tracy\Debugger;
use Tracy\ILogger;

class OrderForm extends BaseControl
{
	final public const sessionNamespace = 'eshopOrdersOrderForm';

	public static array $personalColumns = ['firstName', 'lastName', 'company', 'phone', 'email', 'country',
		'street', 'city', 'postal', 'idNumber', 'vatNumber'];

	protected SessionSection $sessionSection;

	/** @var Spedition[]|null */
	protected ?array $speditions = null;

	/** @var Payment[]|null */
	protected ?array $payments = null;

	/** @var PaymentSpedition[] */
	private ?array $paymentSpeditions = null;

	protected Cart   $cart;
	protected ?array $paymentsSpeditions  = null;
	protected ?array $cCountries          = null;
	protected bool   $allowNewsletterFlag = false;

	public function __construct(
		protected IOrderCartDetailFactory $orderCartDetailFactory,
		protected IPaySpedSummaryFactory  $paySpedSummaryFactory,
		protected IOrderSummaryFactory    $orderSummaryFactory,
		protected Orders                  $ordersService,
		protected CartFacade              $cartFacade,
		protected CartHelper              $cartHelperService,
		protected Speditions              $speditionsService,
		protected Payments                $paymentsService,
		protected PaymentSpeditions       $paymentSpeditionsService,
		protected Customers               $customersService,
		protected Statuses                $statusesService,
		protected Countries               $countriesService,
		protected Session                 $session,
		protected User                    $user,
		protected ProductsFacade          $products,
		protected Invoices                $invoices,
		protected InvoiceConfigRepository $invoiceConfigRepository,
		protected Sites                   $sitesService,
		protected PriceFilter             $priceFilter,
		protected Sellers                 $sellers,
		protected Currencies              $currencies,
		protected Psc                     $pscHelper,
		protected AppState                $appState,
	)
	{
		$this->sessionSection = $session->getSection(self::sessionNamespace);

		$this->cart = $this->cartFacade->getCart();

		if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
			self::$personalColumns[] = 'idVatNumber';
		}

		if (EshopOrdersConfig::load('orderForm.allowNewsletterFlag')) {
			$this->allowNewsletterFlag();
		}
	}

	public function render(): void
	{
		$siteIdent  = $this->sitesService->getCurrentSite()->getIdent();
		$this->cart = $this->cartFacade->getCart();
		$this->template->setFile($this->getTemplateFile());

		$this->template->visibleCountSpedition = EshopOrdersConfig::load('orderForm.orderPage.visibleCountSpedition');
		$this->template->visibleCountPayment   = EshopOrdersConfig::load('orderForm.orderPage.visibleCountPayment');

		$this->template->paymentSpeditions = $this->getPaymentSpeditions();
		$this->template->speditions        = $this->getSpeditions();
		$this->template->payments          = $this->getPayments();
		$this->template->countries         = $this->getCountries();
		$this->template->cart              = $this->cart;

		$form                     = $this['form'];
		$this->template->thisForm = $form;

		$this->template->termsAndConditionsNavId = (int) $this->settings->get(
			'eshopCatalog' . ucfirst($siteIdent) . 'TermsAndConditionsNavId',
			0,
		);
		$this->template->gdprNavId               = (int) $this->settings->get(
			'eshopCatalog' . ucfirst($siteIdent) . 'GdprNavId',
			0,
		);

		$paymentSpeditions                           = $this->getPaymentsSpeditionsCombinations();
		$this->template->paymentSpeditionsStructured = $paymentSpeditions['structured'];
		$this->template->paymentSpeditionsReversed   = $paymentSpeditions['reversed'];

		$this->template->orderData = $this->sessionSection->get('orderFormData');

		$this->template->render();
	}

	public function attached($presenter): void
	{
		parent::attached($presenter);

		$sessionData = $this->sessionSection->get('orderFormData');

		$user = $this->getPresenter()->getUser();
		if ($user->isLoggedIn()) {
			$customer = $this->customersService->getByUser($user->getIdentity());

			if ($customer) {
				$deliAddr = $customer->getAddressDelivery();

				if ($customer && $deliAddr && is_null($sessionData['useAddrDeli']) && $deliAddr->getFirstName() && $deliAddr->getLastName()) {
					$sessionData['useAddrDeli'] = true;
					$this->sessionSection->set('orderFormData', $sessionData);
				}
			}
		}

		if ($sessionData) {
			$this->setOrderData($sessionData);
		}
	}

	/*******************************************************************************************************************
	 * ======================== Handle
	 */

	public function handleSaveStep2()
	{
		$request    = $this->getPresenter()->getHttpRequest();
		$data       = $this->sessionSection->get('orderFormData');
		$speditions = $this->getSpeditions();

		/** @var Spedition|null $sessionSpedition */
		$sessionSpedition = $speditions[$data['speditionCountry']][$data['spedition']] ?? null;

		$data['speditionCountry'] = $request->getPost('speditionCountry');
		foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v) {
			$data[$k] = (int) $request->getPost($v)[$data['speditionCountry']];
		}

		/** @var Spedition|null $spedition */
		$spedition = $speditions[$data['speditionCountry']][$data['spedition']] ?? null;

		if ($spedition && $spedition->isPickup && $spedition->getIdent() === 'online') {
			$data['disableDeliveryAddress'] = true;
			$data['useAddrDeli']            = false;
		} else {
			unset($data['disableDeliveryAddress']);
			unset($data['useAddrDeli']);
		}

		// Pokud menim dopravu z "pickup" na "adresu", tak odskrtnout predvyplnene udaje doruceni
		if ($sessionSpedition && $sessionSpedition->isPickup && $spedition && !$spedition->isPickup) {
			$data['useAddrDeli'] = false;
			$this['form']['useAddrDeli']->setValue(false);
		}

		if ($data['speditionCountry']) {
			$this->appState->setCountry($data['speditionCountry']);
		}

		if (!isset($data['country'])) {
			$this['form']['country']->setValue($data['speditionCountry']);
		}

		if (!isset($data['country2'])) {
			$this['form']['country2']->setValue($data['speditionCountry']);
		}

		$this->eventDispatcher->dispatch(
			new SaveOrderFormDataEvent($data, $this['form']),
			'eshopOrders.saveOrderFormDataStep2',
		);
		$this->sessionSection->set('orderFormData', $data);

		if ($data['disableDeliveryAddressSpedition'] !== null) {
			$this['form']['country2']->setValue($data['speditionCountry']);
		}

		$currencyChange = null;
		if (isset($data['payment'])) {
			$payments = $this->getPayments();
			$payment  = $payments[$data['speditionCountry']][$data['payment']] ?? null;

			if ($payment && $payment->currency && $this->currencies->getCurrent()->getCode() != $payment->currency) {
				$curr = $this->currencies->getActive()[$payment->currency] ?? null;

				if ($curr) {
					$currencyChange = $curr;
				}
			}
		}

		if (!$currencyChange && isset($data['payment'], $data['spedition'])) {
			foreach ($this->getPaymentSpeditions() as $ps) {
				if ($this->currencies->getCurrent()->getCode() != $ps->currency
					&& array_key_exists((string) $data['speditionCountry'], $ps->getCountries())
					&& $ps->getSpedition()->getId() === $data['spedition'] && $ps->getPayment()
						->getId() === $data['payment']) {
					$curr = $this->currencies->getActive()[$ps->currency] ?? null;

					if ($curr) {
						$currencyChange = $curr;
					}

					break;
				}
			}
		}

		if ($currencyChange) {
			$this->currencies->setCurrentCurrency($currencyChange->getId());
			$this->getPresenter()
				->flashMessageInfo(
					$this->t(
						'eshopOrdersFront.orderPage.currencyPaymentChangeTo',
						['curr' => $currencyChange->getCode()],
					),
				);
			$this->getPresenter()->redirect('this');
		}

		$this->redrawControl('step3');
		$this->redrawControl('orderCartDetail');
		$this->redrawControl('orderSummary');
	}

	public function handleSaveStep3()
	{
		$request    = $this->getPresenter()->getHttpRequest();
		$data       = $this->sessionSection->get('orderFormData');
		$form       = $this['form'];
		$formValues = $form->getValues();

		$countries        = $this->countriesService->getAllNameColumn();
		$speditionCountry = $data['speditionCountry'];

		/** @var Customer|null $customer */
		$customer = $this->user->getIdentity() ? $this->customersService->getByUser($this->user->getIdentity()) : null;

		$data['companyOrder'] = $formValues->companyOrder;
		foreach (self::$personalColumns as $k) {
			if (!$customer || !$customer->addressInvoice || !EshopOrdersConfig::load(
					'customer.disableModifyInvoiceAddress',
				)) {
				$data[$k] = $request->getPost($k);
			}

			$data[$k . '2'] = $request->getPost($k . '2');
		}

		$data['deliveryAddressesList'] = $formValues->deliveryAddressesList;

		if ($customer && $customer->addressInvoice && EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
			foreach ($customer->getAddressInvoice() as $k => $v) {
				if ($k === 'country') {
					if (!$v) {
						$data[$k]            = $speditionCountry;
						$data['countryText'] = $countries[$speditionCountry];
					} else {
						/** @var Country $v */
						$data[$k]            = $v->getId();
						$data['countryText'] = $v->getName();
					}
				} else {
					$data[$k] = $v;
				}
			}
		} else {
			$data['invoiceAddressesList'] = $formValues->invoiceAddressesList;
			$data['countryText']          = $countries[$request->getPost('country')] ?? null;
		}

		$data['country2Text'] = $countries[$request->getPost('country2')] ?? null;

		if (!$this->pscHelper->validatePostal((string) $data['postal'], (string) $data['country'])) {
			$form['postal']->addError('eshopOrdersFront.orderPage.badPsc');
		}

		foreach (['useAddrDeli', 'speditionCountry'] as $k) {
			$data[$k] = $request->getPost($k);
		}

		if ($data['useAddrDeli'] === 'on' || $data['useAddrDeli'] === true) {
			$data['useAddrDeli'] = true;

			if ($data['disableDeliveryAddress'] !== true) {
				foreach (['firstName2', 'lastName2', 'email2', 'phone2', 'street2', 'city2', 'postal2'] as $col) {
					if (!$data[$col]) {
						$form[$col]->addError('default.formMessages.filled');
					}
				}

				if (!Validators::isEmail((string) $data['email2'])) {
					$form['email2']->addError('default.formMessages.email');
				}

				if (strlen((string) $data['phone2']) < 9) {
					$form['phone2']->addError('default.formMessages.phone_min');
				} else if (!PhoneInput::validatePhone($form['phone2'])) {
					$form['phone2']->addError(PhoneInput::$phoneMessages);
				}

				if (!$this->pscHelper->validatePostal((string) $data['postal2'], (string) $data['country2'])) {
					$form['postal2']->addError('eshopOrdersFront.orderPage.badPsc');
				}

				if ($data['speditionCountry'] !== $data['country2']) {
					$form['country2']->addError($this->t('eshopOrdersFront.orderPage.badSpeditionCountry'), false);
				}
			}
		} else {
			$data['useAddrDeli'] = false;
			if ($data['country'] != $speditionCountry) {
				$this['form']->addError($this->t('eshopOrdersFront.orderPage.badSpeditionCountry'), false);
			}
		}


		if ($this->isIcoRequired() && $formValues['companyOrder'] === false) {
			$this['form']->addError('eshopOrdersFront.orderPage.companyOrderIsRequired');
		}

		$this->eventDispatcher->dispatch(
			new SaveOrderFormDataEvent($data, $this['form']),
			'eshopOrders.saveOrderFormDataStep3',
		);
		$this['form']['validatePersonal']->validate();
		$this->sessionSection->set('orderFormData', $data);

		if ($this['form']->hasErrors()) {
			$fieldErrors = [];
			foreach (self::$personalColumns as $col) {
				foreach (['', '2'] as $num) {
					if (isset($form[$col . $num]) && $form[$col . $num]->hasErrors()) {
						$fieldErrors[$col . $num] = $form[$col . $num]->getErrors();
					}
				}
			}

			$this->getPresenter()->payload->fieldErrors      = $fieldErrors;
			$this->getPresenter()->payload->orderFormErrors  = $this['form']->getErrors();
			$this->getPresenter()->payload->orderFormIsValid = false;
		}

		$this->redrawControl('step3');
		$this->redrawControl('orderSummary');
	}

	public function isIcoRequired(): bool
	{
		if (EshopOrdersConfig::load('orderForm.icoIsRequired', false)) {
			return true;
		}

		$customer = $this->user->getIdentity() ? $this->customersService->getByUser($this->user->getIdentity()) : null;

		if ($customer && $customer->getGroupCustomers()) {
			$requireIco = $customer->getGroupCustomers()->getParams()['requireIco'] ?? false;

			return EshopOrdersConfig::load('customerGroup.enableRequireIco') && $requireIco;
		}

		return false;
	}

	public function handleSetSpedition($spedition): void
	{
		$data = $this->sessionSection->get('orderFormData');

		$data['spedition'] = (int) $spedition;
		$this->sessionSection->set('orderFormData', $data);
		$this->cartFacade->getCart();
		$this->redrawControl('orderCartDetail');
	}

	public function handleSetPayment($payment): void
	{
		$data = $this->sessionSection->get('orderFormData');

		$data['payment'] = (int) $payment;
		$this->sessionSection->set('orderFormData', $data);
		$this->cartFacade->getCart();
		$this->redrawControl('orderCartDetail');
	}

	public function handleChangeSpeditionCountry(): void
	{
		$this->redrawControl('orderCartDetail');
	}

	/*******************************************************************************************************************
	 * ======================== Setters
	 */
	public function allowNewsletterFlag(bool $allow = true): void { $this->allowNewsletterFlag = $allow; }

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		$form->setAjax();
		$cart      = $this->cartFacade->getCart();
		$countries = $this->countriesService->getAllNameColumn();

		$currency = $this->currencies->getCurrent();
		$form->addHidden('currency', $currency->getCode());
		$form->addGroup('speditions');

		$appStateCountry    = $this->appState->getCountry();
		$speditionCountries = $this->getCountries();
		$defaultCountry     = isset($speditionCountries[$appStateCountry]) ? $appStateCountry : key(
			$speditionCountries,
		);

		$form->addSelect('speditionCountry', 'eshopOrders.orderPage.country', $this->getCountries())
			->setTranslator(null)
			->setDefaultValue($defaultCountry)
			->setHtmlAttribute('data-change-link', $this->link('changeSpeditionCountry!'))
			->setRequired('eshopOrdersFront.orderPage.requireSpeditionCountry');

		$speditionsContainer = $form->addContainer('speditions');
		$paymentsContainer   = $form->addContainer('payments');
		$paySpedCombinations = $this->getPaymentsSpeditionsCombinations();

		if ($this->getPresenter()
				->getUser()
				->isLoggedIn() && EshopOrdersConfig::load('allowAddressManagement', false)) {
			$customer = $this->customersService->getByUserId($this->getPresenter()->getUser()->getId());
			if ($customer) {
				$addresses    = [];
				$addressesArr = [];
				$icoData      = [
					'idNumber'  => null,
					'vatNumber' => null,
				];
				$addrToModify = [];

				/** @var CustomerAddress[] $addressesList */
				$addressesList = $customer->addressesAvailable->toArray();

				foreach ($addressesList as $addr) {
					if ($customer->addressInvoice && $customer->addressInvoice->getId() === $addr->getId()) {
						continue;
					}

					$addresses[$addr->getId()]    = $addr->getDetailString();
					$addressesArr[$addr->getId()] = $addr->toArray();

					if (EshopOrdersConfig::load('customer.disableModifyIco')) {
						foreach ($icoData as $colKey => $colVal) {
							if (!$colVal && $addressesArr[$addr->getId()][$colKey]) {
								$icoData[$colKey]             = $addressesArr[$addr->getId()][$colKey];
								$addrToModify[$addr->getId()] = $addr->getId();
							}
						}
					}
				}

				if (EshopOrdersConfig::load('customer.disableModifyIco') && !empty($addrToModify)) {
					foreach ($addressesArr as $id => $addr) {
						foreach ($icoData as $colKey => $colVal) {
							if ($colVal && !$addr[$colKey]) {
								$addressesArr[$id][$colKey]    = $colVal;
								$addressesList[$id]->{$colKey} = $colVal;
							}
						}

						$this->em->persist($addressesList[$id]);
						$this->em->flush($addressesList[$id]);
					}
				}

				$addressesArr = Json::encode($addressesArr);

				$form->addSelect(
					'deliveryAddressesList',
					'eshopOrdersFront.orderPage.selectDeliveryAddress',
					$addresses,
				)
					->setHtmlAttribute('data-list', $addressesArr)
					->setHtmlAttribute('data-name-suffix', '2');

				if ($customer->getAddressDelivery()) {
					$form['deliveryAddressesList']->setDefaultValue($customer->getAddressDelivery()->getId());
				}

				if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					$form->addSelect(
						'invoiceAddressesList',
						'eshopOrdersFront.orderPage.selectInvoiceAddress',
						$addresses,
					)
						->setHtmlAttribute('data-list', $addressesArr)
						->setHtmlAttribute('data-name-suffix', '');

					if ($customer->getAddressInvoice()) {
						$form['invoiceAddressesList']->setDefaultValue($customer->getAddressInvoice()->getId());
					}
				}
			}
		}

		foreach ($this->getSpeditions() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['structured'][$country][$row->getId()]))
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format(
							(float) $row->getPrice(),
						);
			}

			$speditionsContainer->addRadioList($country, 'eshopOrders.orderPage.spedition', $list)
				->addConditionOn($form['speditionCountry'], BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED, 'eshopOrdersFront.orderPage.requireSpedition');
		}

		foreach ($this->getPayments() as $country => $rows) {
			$list = [];

			foreach ($rows as $row) {
				if (isset($paySpedCombinations['reversed'][$country][$row->getId()])) {
					$list[$row->getId()] = $row->getName() . ' - ' . $this->priceFilter->format(
							(float) $row->getPrice(),
						);
				}
			}

			$paymentsContainer->addRadioList($country, 'eshopOrders.orderPage.payment', $list)
				->addConditionOn($form['speditionCountry'], BaseForm::EQUAL, $country)
				->addRule(BaseForm::REQUIRED, 'eshopOrdersFront.orderPage.requirePayment');
		}

		$form->addSubmit('validateSpedition', 'eshopOrders.orderPage.nextStep')
			->setValidationScope([
				$form['speditionCountry'],
				$form['payments'],
				$form['speditions'],
			])->onClick[] = $this->handleSaveStep2(...);

		$form->addGroup('personal');
		$form->addText('firstName', 'eshopOrders.orderPage.name')->setRequired();
		$form->addText('lastName', 'eshopOrders.orderPage.lastName')->setRequired();
		$form->addPhone('phone', 'eshopOrders.orderPage.phone')
			->addRule(PhoneInput::PHONE, PhoneInput::$phoneMessages)->setRequired();
		$form->addEmail('email', 'eshopOrders.orderPage.email')->setRequired();
		$form->addText('street', 'eshopOrders.orderPage.street')->setRequired();
		$form->addText('city', 'eshopOrders.orderPage.city')->setRequired();
		$form->addText('postal', 'eshopOrders.orderPage.postal')
			->setRequired();
		$form->addSelect('country', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);
		$form->addCheckbox('companyOrder', 'eshopOrdersForm.orderPage.companyOrder');
		$form->addText('idNumber', 'eshopOrders.orderPage.idNumber')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('company', 'eshopOrders.orderPage.company')
			->addConditionOn($form['companyOrder'], BaseForm::EQUAL, true)->setRequired();
		$form->addText('vatNumber', 'eshopOrders.orderPage.vatNumber');

		if ($this->getPresenter()->getUser()->isLoggedIn() && EshopOrdersConfig::load('customer.disableModifyIco')) {
			$form['idNumber']->setDisabled();
			$form['vatNumber']->setDisabled();
		}

		if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
			$form->addText('idVatNumber', 'eshopOrders.orderPage.idVatNumber');
		}

		if ($this->isIcoRequired()) {
			$form['companyOrder']->setDefaultValue(true)->setRequired();
		}

		if (EshopOrdersConfig::load('orderForm.allowLoadFromAres')) {
			$form['idNumber']->setDescription(
				'<a href="javascript:;" class="btn secondary-bg-color" data-order-form-ico-load="' . Parameters::load(
					'api.urls.loadIco',
				) . '" data-error-text="' . $this->t('eshopOrdersFront.orderPage.icoLoadError') . '">' . $this->t(
					'eshopOrdersFront.orderPage.autoFillByIco',
				) . '</a>',
			);
		}

		if (EshopOrdersConfig::load('orderForm.autocompletePsc')) {
			$form['postal']
				->setHtmlAttribute('data-autocomplete-url', EshopOrdersConfig::load('api.pscUrl'))
				->setHtmlAttribute('data-autocomplete-target', $form['postal']->getHtmlId())
				->setHtmlAttribute('data-autocomplete-key', 'psc')
				->setHtmlAttribute('data-result-item-data', 'psc');
		}

		//fakturacni adresa
		$useAddrDeli = $form->addCheckbox('useAddrDeli', 'eshopOrders.orderPage.useAddrDeli');
		$form->addText('company2', 'eshopOrders.orderPage.company');
		$form->addText('firstName2', 'eshopOrders.orderPage.name');
		$form->addText('lastName2', 'eshopOrders.orderPage.lastName');
		$form->addText('email2', 'eshopOrders.orderPage.email');
		$form->addPhone('phone2', 'eshopOrders.orderPage.phone');
		$form->addText('street2', 'eshopOrders.orderPage.street');
		$form->addText('city2', 'eshopOrders.orderPage.city');
		$form->addText('postal2', 'eshopOrders.orderPage.postal');

		$form->addSelect('country2', 'eshopOrdersFront.orderPage.country', $countries)
			->setTranslator(null);

		if ($firstNameMaxLength = EshopOrdersConfig::load('orderForm.fields.firstName.maxLength')) {
			$form['firstName']->addRule($form::MAX_LENGTH, null, $firstNameMaxLength);
			$form['firstName2']->addRule($form::MAX_LENGTH, null, $firstNameMaxLength);
		}

		if ($lastNameMaxLength = EshopOrdersConfig::load('orderForm.fields.lastName.maxLength')) {
			$form['lastName']->addRule($form::MAX_LENGTH, null, $lastNameMaxLength);
			$form['lastName2']->addRule($form::MAX_LENGTH, null, $lastNameMaxLength);
		}

		if ($streetMaxLength = EshopOrdersConfig::load('orderForm.fields.street.maxLength')) {
			$form['street']->addRule($form::MAX_LENGTH, null, $streetMaxLength);
			$form['street2']->addRule($form::MAX_LENGTH, null, $streetMaxLength);
		}

		if ($cityMaxLength = EshopOrdersConfig::load('orderForm.fields.city.maxLength')) {
			$form['city']->addRule($form::MAX_LENGTH, null, $cityMaxLength);
			$form['city2']->addRule($form::MAX_LENGTH, null, $cityMaxLength);
		}

		if (EshopOrdersConfig::load('orderForm.autocompletePsc')) {
			$form['postal2']
				->setHtmlAttribute('data-autocomplete-url', EshopOrdersConfig::load('api.pscUrl'))
				->setHtmlAttribute('data-autocomplete-target', $form['postal2']->getHtmlId())
				->setHtmlAttribute('data-autocomplete-key', 'psc')
				->setHtmlAttribute('data-result-item-data', 'psc');
		}


		$form['country']->addRule(function($item, BaseForm $args) {
			$data   = $this->sessionSection->get('orderFormData');
			$values = $args->getUntrustedValues();

			if ($values->useAddrDeli === false) {
				return true;
			} else if ($values->useAddrDeli === true) {
				return true;
			} else {
				return $values['speditionCountry'] === $item->getValue();
			}
		}, 'eshopOrdersFront.orderPage.badSpeditionCountry', $form);

		$validateFields = [];
		foreach (self::$personalColumns as $v) {
			$validateFields[] = $form[$v];

			if (!in_array($v, ['idNumber', 'vatNumber']) && isset($form[$v . '2'])) {
				$validateFields[] = $form[$v . '2'];
			}
		}

		$form->addSubmit('validatePersonal', 'eshopOrders.orderPage.nextStep')
			->setValidationScope($validateFields)
			->onClick[] = $this->handleSaveStep3(...);

		$form->addTextArea('messageAlt', 'eshopOrders.orderPage.message')->setAttribute('class', ['messageInput']);
		$form->addTextArea('message', 'eshopOrders.orderPage.message');

		if (EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false)) {
			$form->addCheckbox('agreedOne', 'eshopOrders.orderPage.agreedTerms')
				->setDefaultValue(1);
			if (EshopOrdersConfig::load('orderForm.requiredGdprOneCheckbox', false)) {
				$form['agreedOne']->setRequired('eshopOrders.orderPage.agreedTermsRequired');
			}
		} else {
			$form->addCheckbox('agreedTerms', 'eshopOrders.orderPage.agreedTerms')
				->setRequired('eshopOrders.orderPage.agreedTermsRequired');

			if (EshopOrdersConfig::load('orderForm.agreedTermsAlwaysChecked', false)) {
				$form['agreedTerms']->setRequired(false)
					->setDefaultValue(1);
			}

			if (EshopOrdersConfig::load('orderForm.agreedQuestioning', true)) {
				$form->addCheckbox('agreedQuestioning', 'eshopOrdersFront.orderPage.agreedQuestioning')
					->setDefaultValue(1);

				if (EshopOrdersConfig::load('orderForm.agreedQuestioningInvert', false)) {
					$form['agreedQuestioning']->setCaption('eshopOrdersFront.orderPage.agreedQuestioningInvert')
						->setDefaultValue(0);
				}
			} else {
				$form->addHidden('agreedQuestioning', 0);
			}
		}

		if ($this->allowNewsletterFlag) {
			$form->addCheckbox('agreedNewsletter', 'eshopOrdersFront.orderPage.agreedNewsletter')
				->setDefaultValue(1);

			if (EshopOrdersConfig::load('orderForm.allowNewsletterFlagInvert', false)) {
				$form['agreedNewsletter']->setCaption('eshopOrdersFront.orderPage.agreedNewsletterInvert')
					->setDefaultValue(0);
			}
		}

		$form->addSubmit('submit', 'eshopOrders.orderPage.send')
			->setHtmlAttribute('class', ['submitButton', 'eshopOrdersNext4'])
			->setHtmlId('orderFormSubmit');

		if ($this->user->getIdentity()) {
			$customer = $this->customersService->getByUser($this->user->getIdentity());

			if ($customer) {
				$addrDeli = $customer->getAddressDelivery();
				$addrInvo = $customer->getAddressInvoice();

				if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					$form['companyOrder']->setDisabled();
					$form['idNumber']->setDescription('');

					foreach (self::$personalColumns as $col) {
						$form[$col]->setDisabled();
					}
				}

				if ($addrInvo) {
					$form->setDefaults($addrInvo->toArray());

					if ($addrInvo->getIdNumber()) {
						$d = [
							'companyOrder' => true,
							'company'      => $addrInvo->getCompany(),
							'idNumber'     => $addrInvo->getIdNumber(),
							'vatNumber'    => $addrInvo->getVatNumber(),
						];

						if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
							$d['idVatNumber'] = $addrInvo->idVatNumber;
						}

						$form->setDefaults($d);

						$this->template->companyFilled = true;
					}
				}

				if ($addrDeli) {
					$def = [];
					foreach ($addrDeli->toArray() as $k => $v)
						$def[$k . '2'] = $v;
					$form->setDefaults($def);
				}
			}
		}

		if ($this->isIcoRequired()) {
			$form['companyOrder']->setValue(true);
			$form['idNumber']->setRequired();

			$this->template->companyFilled = true;
		}

		$form->onValidate[] = $this->formOnValidate(...);
		$form->onSuccess[]  = $this->formOnSuccess(...);

		$orderFormEvent           = new OrderFormEvent($form);
		$orderFormEvent->template = $this->template;
		$this->eventDispatcher->dispatch($orderFormEvent, 'eshopOrders.createOrderForm');

		return $form;
	}

	public function setOrderData(array $orderData): void
	{
		$this->eventDispatcher->dispatch(
			new SetOrderFormDataEvent($orderData, $this['form']),
			'eshopOrders.setOrderFormData',
		);

		/** @var Customer|null $customer */
		$customer = $this->getPresenter()->user->isLoggedIn() && $this->user->getIdentity()
			? $this->customersService->getByUser($this->user->getIdentity())
			: null;

		$form = $this['form'];
		$d    = [
			'messageAlt'       => $orderData['messageAlt'],
			'message'          => $orderData['message'],
			'useAddrDeli'      => $orderData['useAddrDeli'],
			'agreedNewsletter' => $orderData['agreedNewsletter'],
		];

		foreach (['deliveryAddressesList', 'invoiceAddressesList'] as $v) {
			if ($orderData[$v])
				$d[$v] = $orderData[$v];
		}

		if (EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false)) {
			$d['agreedOne'] = $orderData['agreedOne'];
		} else {
			if (EshopOrdersConfig::load('orderForm.agreedTermsAlwaysChecked', false)) {
				$orderData['agreedTerms'] = 1;
			}

			$d['agreedTerms']       = $orderData['agreedTerms'];
			$d['agreedQuestioning'] = $orderData['agreedQuestioning'];
		}

		$spedCountry = $orderData['speditionCountry'];
		$countries   = $this->countriesService->getAllNameColumn();
		if ($spedCountry) {
			if (array_key_exists($spedCountry, $form['speditionCountry']->getItems())) {
				$d['speditionCountry'] = $spedCountry;
			}

			foreach (['spedition' => 'speditions', 'payment' => 'payments'] as $k => $v) {
				if ($orderData[$k] && isset($form[$v][$spedCountry]) && array_key_exists(
						$orderData[$k],
						$form[$v][$spedCountry]->getItems(),
					)) {
					$d[$k][$spedCountry] = $orderData[$k];
				}
			}
		}

		foreach (self::$personalColumns as $k) {
			if (!$customer || !$customer->getAddressInvoice() || !EshopOrdersConfig::load(
					'customer.disableModifyInvoiceAddress',
				)) {
				if ($orderData[$k]) {
					$d[$k] = $orderData[$k];
				}
			} else {
				foreach ($customer->getAddressInvoice() as $ak => $av) {
					if ($ak === 'country') {
						if (!$av) {
							$d[$k]            = $spedCountry;
							$d['countryText'] = $countries[$spedCountry];
						} else {
							/** @var Country $av */
							$d[$ak]           = $av->getId();
							$d['countryText'] = $av->getName();
						}
					} else {
						$d[$ak] = $av;
					}
				}
			}

			if (!in_array($k, ['idNumber', 'vatNumber']) && $orderData[$k . '2']) {
				$d[$k . '2'] = $orderData[$k . '2'];
			}
		}

		$form->setDefaults($d);
	}

	public function formOnValidate(BaseForm $form, ArrayHash $values): bool
	{
		$data                                = $this->sessionSection->get('orderFormData');
		$cart                                = $this->cartFacade->getCart();
		$cartItems                           = $cart->getCartItems();
		$this->template->formSummaryMessages = [];

		if (empty($cartItems) && EshopOrdersConfig::load('orderForm.enableEmptyCartValidation', true)) {
			$form->addError('eshopOrders.orderPage.emptyCart');
			$this->template->formSummaryMessages = ['eshopOrders.orderPage.emptyCartData'];
		}

		$spedCountry = $values->speditionCountry;
		if ($spedCountry) {
			$spedition = $values->speditions[$spedCountry] ?? null;
			$payment   = $values->payments[$spedCountry] ?? null;

			$paymentsSpeditions = $this->getPaymentsSpeditionsCombinations();
			$paySped            = $paymentsSpeditions['structured'][$spedCountry][$spedition] ?? null;
			if (!$paySped || !in_array($payment, $paySped)) {
				$form->addError('eshopOrdersFront.orderPage.invalidPaymentSpeditionCombination');
				Debugger::log(
					'payment: ' . $payment . ', spedition: ' . $spedition . ', country: ' . $spedCountry,
					'paymentSpeditionCombination',
				);
			}
		}

		$remainingValue = $cart->getRemainingValueToCompleteOrder();
		if ($remainingValue > 0) {
			$form->addError(
				Html::el()
					->setHtml(
						$this->t(
							'eshopOrdersFront.orderPage.youNeedMoreToCompleteOrder',
							['price' => $this->priceFilter->format($remainingValue)],
						),
					),
				false,
			);
		}

		if (Floats::isLessThan($cart->getPriceTotal(), 0.0)) {
			$form->addError('eshopOrdersFront.orderPage.invalidOrderPrice');
		}

		$this->eventDispatcher->dispatch(new OrderFormEvent($form), 'eshopOrders.orderFormValidate');

		if (!empty($form->getErrors())) {
			Debugger::log($form->getErrors(), 'orderFormValidate');
		}

		if ($form->hasErrors()) {
			$errs = [];
			foreach ($form->getComponents(true) as $c) {
				if ($c instanceof BaseContainer) {
					continue;
				}

				if ($c->getErrors()) {
					$errs[$c->getHtmlId()] = $c->getErrors();
				}
			}

			$this->getPresenter()->payload->orderFormErrors  = $errs;
			$this->getPresenter()->payload->orderFormIsValid = false;

			foreach ($form->getErrors() as $e) {
				$this->template->formSummaryMessages[] = $e;
			}

			$this->redrawControl('formMessages');

			$this->getPresenter()->redrawControl('cartMessage');
			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');

			return false;
		} else {
			$this->getPresenter()->payload->orderFormIsValid = true;

			$this->getPresenter()->redrawControl('cartMessage');
			$this->redrawControl('formErrors2');
			$this->redrawControl('formErrors3');
			$this->redrawControl('formErrors4');
		}

		return true;
		// nebude potreba take validovat jestli doprava a platba jsou aktivni a plati pro cenu kosiku?
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$orderFlagQuestioning = null;
		if (!$form['submit']->isSubmittedBy()) {
			return false;
		}

		try {
			$orderFormData = $this->sessionSection->get('orderFormData');

			/** @var Site $currentSite */
			$currentSite = $this->em->getReference(Site::class, $this->sitesService->getCurrentSite()->getIdent());
			$order       = new Order($currentSite);
			$order->lang = $this->translator->getLocale();

			$agreedQuestioning = $values->agreedQuestioning;
			$agreedNewsletter  = $values->agreedNewsletter;

			if (EshopOrdersConfig::load('orderForm.agreedQuestioningInvert', false)) {
				$agreedQuestioning = !$agreedQuestioning;
			}

			if (EshopOrdersConfig::load('orderForm.allowNewsletterFlagInvert', false)) {
				$agreedNewsletter = !$agreedNewsletter;
			}

			if ($values->speditionCountry && isset(
					EshopOrdersConfig::load(
						'order.countryLangs',
					)[$values->speditionCountry],
				)) {
				$order->lang = EshopOrdersConfig::load('order.countryLangs')[$values->speditionCountry];
			}

			$order->setMessage($values->message ?: $values->messageAlt);
			$order->setAgreedTerms(
				$values->agreedTerms
				|| $values->agreedOne
				|| EshopOrdersConfig::load('orderForm.agreedTermsAlwaysChecked', false),
			);

			if ($agreedQuestioning
				|| ($values->agreedOne && EshopOrdersConfig::load('orderForm.showOnlyOneCheckbox', false))
			) {
				$orderFlagQuestioning = new OrderFlag(OrderFlag::TYPE_QUESTIONING, true, $order);
				$this->em->persist($orderFlagQuestioning);
				$order->addFlag($orderFlagQuestioning);
			}

			if ($agreedNewsletter) {
				$orderFlagNewsletter = new OrderFlag(OrderFlag::TYPE_NEWSLETTER, true, $order);
				$this->em->persist($orderFlagNewsletter);
				$order->addFlag($orderFlagNewsletter);
			}

			/** @var Customer|null $customer */
			$customer = $this->getPresenter()->user->isLoggedIn() && $this->user->getIdentity()
				? $this->customersService->getByUser($this->user->getIdentity())
				: null;
			if ($customer && $customer->getAddressInvoice() && EshopOrdersConfig::load(
					'customer.disableModifyInvoiceAddress',
				)) {
				$tmp     = $customer->getAddressInvoice();
				$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
				$addrInv->setFirstName($tmp->firstName);
				$addrInv->setLastName($tmp->lastName);
				$addrInv->setEmail($tmp->email);
				$addrInv->setPhone($tmp->phone);
				$addrInv->setStreet($tmp->street);
				$addrInv->setCity($tmp->city);
				$addrInv->setPostal($tmp->postal);
				if ($tmp->getCountry()) {
					$addrInv->setCountry($tmp->getCountry());
				} else {
					$addrInv->setCountry($this->countriesService->getReference($orderFormData['country']));
				}

				if ($tmp->company) {
					$addrInv->setCompany($tmp->company);
				}
				if ($tmp->idNumber) {
					$addrInv->setIdNumber($tmp->idNumber);
				}
				if ($tmp->vatNumber) {
					$addrInv->setVatNumber($tmp->vatNumber);
				}
				$addrInv->idVatNumber = $tmp->idVatNumber ?: null;
			} else {
				$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
				$addrInv->setFirstName($values->firstName);
				$addrInv->setLastName($values->lastName);
				$addrInv->setEmail($values->email);
				$addrInv->setPhone($values->phone);
				$addrInv->setStreet($values->street);
				$addrInv->setCity($values->city);
				$addrInv->setPostal($values->postal);
				if ($values->country) {
					$addrInv->setCountry($this->countriesService->getReference($values->country));
				}

				if ($values->company)
					$addrInv->setCompany($values->company);
				if ($values->idNumber)
					$addrInv->setIdNumber($values->idNumber);
				if ($values->vatNumber)
					$addrInv->setVatNumber($values->vatNumber);
				$addrInv->idVatNumber = $values->idVatNumber ?: null;
			}

			$this->em->persist($addrInv);

			$order->setAddressInvoice($addrInv);

			$addrDeli = null;
			if ($values->useAddrDeli) {
				$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
				$addrDeli->setFirstName($values->firstName2);
				$addrDeli->setLastName($values->lastName2);
				$addrDeli->setEmail($values->email2);
				$addrDeli->setPhone($values->phone2);
				$addrDeli->setStreet($values->street2);
				$addrDeli->setCity($values->city2);
				$addrDeli->setPostal($values->postal2);
				if ($values->country2)
					$addrDeli->setCountry($this->countriesService->getReference($values->country2));
				$addrDeli->setCompany($values->company2);
				$this->em->persist($addrDeli);

				$order->setAddressDelivery($addrDeli);
			}

			$customer = $this->getOrCreateCustomer(
				$order->getAddressInvoice()->getEmail(),
				$order->getAddressInvoice()->getFirstName(),
				$order->getAddressInvoice()->getLastName(),
				$order->getAddressInvoice()->getPhone(),
			);

			if ($customer) {
				$order->setCustomer($customer);

				if (EshopOrdersConfig::load('allowAddressManagement', false)) {
					$customerDeli = $customer->addressesAvailable->get($values->deliveryAddressesList);
					$customerInvo = EshopOrdersConfig::load(
						'customer.disableModifyInvoiceAddress',
					) && $customer->addressInvoice
						? $customer->addressInvoice
						: $customer->addressesAvailable->get($values->invoiceAddressesList);
				} else {
					$customerDeli = $customer->getAddressDelivery();
					$customerInvo = $customer->getAddressInvoice();
				}

				if ($addrInv) {
					if (!$customerInvo) {
						$customerInvo = new CustomerAddress($customer);
					}

					if (!$customerInvo->firstName || !EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
						$customerInvo->fillFromOrderAddress($addrInv);
						$customer->setAddressInvoice($customerInvo);
						$this->em->persist($customerInvo);
					}
				}

				if ($addrDeli) {
					if (!$customerDeli) {
						$customerDeli = new CustomerAddress($customer);
					}

					$customerDeli->fillFromOrderAddress($addrDeli);
					$customer->setAddressDelivery($customerDeli);
					$this->em->persist($customerDeli);
				}

				if (EshopOrdersConfig::load('order.autoValidateIfUserIsValid')) {
					if (EshopOrdersConfig::load('order.allowValidateIdNumber')) {
						if ($addrInv && $addrInv->getIdNumber())
							$addrInv->validatedIdNumber = $customer->getAddressInvoice()->validatedIdNumber;
						if ($addrDeli && $addrDeli->getIdNumber())
							$addrDeli->validatedIdNumber = $customer->getAddressDelivery()->validatedIdNumber;
					}
					if (EshopOrdersConfig::load('order.allowValidateVatNumber')) {
						if ($addrInv && $addrInv->getVatNumber())
							$addrInv->validatedVatNumber = $customer->getAddressInvoice()->validatedVatNumber;
						if ($addrDeli && $addrDeli->getVatNumber())
							$addrDeli->validatedVatNumber = $customer->getAddressDelivery()->validatedVatNumber;
					}

					if ($addrInv)
						$this->em->persist($addrInv);
					if ($addrDeli)
						$this->em->persist($addrDeli);
				}
			}

			$cart       = $this->cartFacade->getCart();
			$cartItems  = $cart->getCartItems();
			$orderItems = $this->ordersService->fillOrderItems($cartItems, $order);
			$order->setOrderItems($orderItems);

			$orderPayment = new OrderPayment(
				$this->paymentsService->getReference((int) $values->payments[$values->speditionCountry]), $order
			);
			$payment      = $this->paymentsService->get((int) $values->payments[$values->speditionCountry]);
			$orderPayment->setVatRate(
				OrderHelper::checkCountryVatRate(
					(int) $cart->payment->vat,
					$addrInv->getCountry()->getId(),
					false,
					$addrInv->getIdNumber(),
					$addrInv->getVatNumber(),
				),
			);
			$orderPayment->setPrice($payment->getPriceActual($cart, true));
			$order->setPayment($orderPayment);

			$orderSpedition = new OrderSpedition(
				$this->speditionsService->getReference((int) $values->speditions[$values->speditionCountry]), $order
			);
			$spedition      = $this->speditionsService->get((int) $values->speditions[$values->speditionCountry]);
			$orderSpedition->setVatRate(
				OrderHelper::checkCountryVatRate(
					(int) $cart->spedition->vat,
					$addrInv->getCountry()->getId(),
					false,
					$addrInv->getIdNumber(),
					$addrInv->getVatNumber(),
				),
			);
			$orderSpedition->setPrice($spedition->getPriceActual($cart, true));
			$order->setSpedition($orderSpedition);

			$statusCreated      = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);
			$order->getOrderStatuses()->add($orderActionCreated);

			$cartOrderGift = $cart->getOrderGift();
			if ($cartOrderGift) {
				$orderGift        = new OrderGift(
					$order,
					$this->em->getReference(Product::class, $cartOrderGift->productId),
					$cartOrderGift->name
				);
				$orderGift->ean   = $cartOrderGift->ean;
				$orderGift->code1 = $cartOrderGift->code1;

				$this->em->persist($orderGift);
			}

			//			foreach ($cart->getDiscounts() as $k => $v) {
			//				$discount = new OrderDiscount($v->id, $order);
			//				$discount->setValue($v->amount);
			//				$discount->setType($v->type);
			//				$discount->setPrice($cart->calculateDiscount($v));
			//				$discount->setName($this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol);
			//				$this->em->persist($discount);
			//				$order->addOrderDiscount($discount);
			//			}

			$this->em->getConnection()->beginTransaction();
			try {
				$eventOrder                 = new OrderEvent($order);
				$eventOrder->addrDelivery   = $addrDeli;
				$eventOrder->addrInvoice    = $addrInv;
				$eventOrder->orderSpedition = $orderSpedition;
				$eventOrder->orderPayment   = $orderPayment;
				$eventOrder->orderItems     = $orderItems;
				$eventOrder->formData       = $values;
				$this->eventDispatcher->dispatch($eventOrder, 'eshopOrders.orderBeforeSave');
				//ulozit objednavku
				$this->em->persist($order);
				//ulozit k ni adresy
				$this->em->persist($addrInv);
				if ($addrDeli)
					$this->em->persist($addrDeli);
				//polozky doprava a platba
				$this->em->persist($orderSpedition);
				$this->em->persist($orderPayment);
				//ulozit k ni polozky
				foreach ($orderItems as $oi) {
					$this->em->persist($oi);
				}
				//akce - datum kdy byla objednavka vytvorena
				$this->em->persist($orderActionCreated);
				//pripadny slevovy kod
				if (isset($discount) && $discount)
					$this->em->persist($discount);
				//pripadne nastavene priznaky
				if ($orderFlagQuestioning)
					$this->em->persist($orderFlagQuestioning);
				// vytvoreni faktury
				if (isset($invoice) && $invoice)
					$this->em->persist($invoice);
				//teprve po ulozeni vseho flush (aby byly spravne propojene cizi klice)
				$this->em->flush();

				$this->ordersService->moveCartFilesToOrder($cartItems, $orderItems);
				$this->em->getConnection()->commit();
				//TODO nejake promazavani kosiku z DB, kdyz uz k nim neni aktivni session... nebo po nejakem timeoutu (tyden a pod)
			} catch (Exception $e) {
				$this->em->getConnection()->rollBack();
				Debugger::log($e->getMessage(), ILogger::ERROR);
				Debugger::log($e, ILogger::ERROR);
				throw $e;
			}

			//dale uz pracujeme s objednavkou nactenou z DB
			$orderDetail = $this->ordersService->get($order->getId());
			unset($order);
			$this->cartFacade->getCartsService()->deleteCurrentCart();
			$this->sessionSection->set('orderFormData', []);

			// Spuštění události po dokončení objednávky jako odeslání emailu, zboží,...
			$this->eventDispatcher->dispatch(new OrderEvent($orderDetail), 'eshopOrders.orderOnSuccess');
			// Vytáhnutí zpráv z událostí
			$messages = $this->session->getSection('eshopOrders/orderSuccessMessages');

			if ($messages->get('success')) {
				foreach ($messages->get('success') as $v) {
					$this->presenter->flashMessageSuccess($v);
				}
			}

			if ($messages->get('error')) {
				foreach ($messages->get('error') as $v) {
					$this->presenter->flashMessageDanger($v);
				}
			}

			$messages->remove();
		} catch (Exception $e) {
			Debugger::log($e, 'exception');
			$form->addError($e->getMessage());

			return false;
		}

		$this->presenter->redirect(
			':EshopOrders:Front:Finished:orderFinished',
			['orderIdent' => $orderDetail->getIdent()],
		);

		return true;
	}

	protected function getOrCreateCustomer($email, $firstName = '', $lastName = '', $phone = '')
	{
		$userId = null;
		if ($this->getPresenter()->getUser()->isLoggedIn()) {
			$userId = $this->getPresenter()->getUser()->id;
		}
		$customer = $this->customersService->getOrCreateCustomer($userId, $email, $firstName, $lastName, $phone);

		return $customer;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderCartDetail()
	{
		$control = $this->orderCartDetailFactory->create();

		return $control;
	}

	protected function createComponentPaySpedSummary()
	{
		$control = $this->paySpedSummaryFactory->create();

		$orderData = $this->sessionSection->get('orderFormData');
		$control->setParameters($orderData['spedition'], $orderData['payment'], $orderData['priceTotal']);

		return $control;
	}

	protected function createComponentOrderSummary()
	{
		$control = $this->orderSummaryFactory->create();

		$orderData = $this->sessionSection->get('orderFormData');
		$control->setParameters($orderData);

		return $control;
	}

	/**
	 * @return Payment[]
	 */
	public function getSpeditions(): array
	{
		if ($this->speditions === null) {
			/** @var Spedition[] $raw */
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getSpedition()->getId()] = $ps->getSpedition()->getId();
				}

				$raw[$ps->getSpedition()->getId()] = $ps->getSpedition();
			}
			$raw = $this->speditionsService->filterByCart($raw, $this->cart);

			$this->speditions = $this->processPaySpedData($byCountry, $raw);

			$this->eventDispatcher->dispatch(
				new Event(['speditions' => &$this->speditions]),
				OrderForm::class . '::loadSpeditions',
			);
		}

		return $this->speditions;
	}

	public function getPaymentsSpeditionsCombinations(): array
	{
		if ($this->paymentsSpeditions === null) {
			$filteredSpedition = $this->getSpeditions();
			$filteredPayments  = $this->getPayments();

			$paymentSpeditions        = $this->getPaymentSpeditions();
			$this->paymentsSpeditions = [
				'structured' => [],
				'reversed'   => [],
			];

			foreach ($paymentSpeditions as $key => $item) {
				foreach ($item->getCountries() as $country) {
					/** @var Country $country */
					$spedId    = $item->getSpedition()->getId();
					$payId     = $item->getPayment()->getId();
					$countryId = $country->getId();

					if (!isset($filteredPayments[$countryId][$payId]) || !isset($filteredSpedition[$countryId][$spedId]))
						continue;

					$this->paymentsSpeditions['structured'][$countryId][$spedId][] = $payId;
					$this->paymentsSpeditions['reversed'][$countryId][$payId][]    = $spedId;
				}
			}
		}

		return $this->paymentsSpeditions;
	}

	/**
	 * @return Payment[][]
	 */
	public function getPayments(): array
	{
		if ($this->payments === null) {
			$raw       = [];
			$byCountry = [];

			foreach ($this->getPaymentSpeditions() as $ps) {
				foreach ($ps->getCountries() as $country) {
					$byCountry[$country->getId()][$ps->getPayment()->getId()] = $ps->getPayment()->getId();
				}

				$raw[$ps->getPayment()->getId()] = $ps->getPayment();
			}
			$raw = $this->paymentsService->filterByCartValue($raw, $this->cart->getCartItemsPrice());

			$this->payments = $this->processPaySpedData($byCountry, $raw);
		}

		return $this->payments;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getPaymentSpeditions(): array
	{
		if ($this->paymentSpeditions === null)
			$this->paymentSpeditions = $this->paymentSpeditionsService->getAllPublished();

		return $this->paymentSpeditions;
	}

	public function getCountries(): array
	{
		if ($this->cCountries === null) {
			$this->cCountries = [];

			foreach ($this->getPaymentSpeditions() as $row)
				foreach ($row->getCountries() as $id => $country)
					$this->cCountries[$country->getId()] = $country->getName();
		}

		return $this->cCountries;
	}

	protected function processPaySpedData(array $byCountry, array $raw)
	{
		$result = [];

		foreach ($byCountry as $country => $rows) {
			foreach ($rows as $row) {
				if (!isset($raw[$row]))
					continue;

				$raw[$row]->country     = $country;
				$result[$country][$row] = $raw[$row];
			}
		}

		return $result;
	}
}
