<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Components\Flashes\Flashes;
use Core\Model\Application\AppState;
use Core\Model\BotDetect;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Module;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\PersistentCollection;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopOrders\FrontModule\Model\Event\FillDaoEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\Model\Entities\Cart;
use EshopOrders\Model\Entities\CartItem;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Exception;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\SmartObject;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Tracy\Debugger;
use Users\Model\Security\User;
use function str_starts_with;

class Carts extends BaseFrontEntityService
{
	use SmartObject;

	protected $entityClass = Cart::class;

	/** @var Dao\Cart[]|null */
	protected $cartsCached;

	protected ?SessionSection $sessionSection = null;
	/** @var array Cart[] */
	protected array $cRawCart = [];

	/** @var Dao\CartItem[] */
	public $cDaoItems;

	public function __construct(
		protected Session          $session,
		protected ProductsFacade   $productsFacadeService,
		protected Tags             $tagsService,
		protected EventDispatcher  $eventDispatcher,
		protected Customers        $customers,
		protected Exchange         $exchange,
		protected User             $user,
		public Currencies          $currencies,
		protected EshopOrdersCache $eshopOrdersCache,
		protected Sites            $sites,
	)
	{
	}

	/**
	 *
	 * @param int $itemId
	 */
	public function getCartItem($itemId): ?Dao\CartItem
	{
		return $this->getCurrentCart()->getCartItems()[$itemId] ?? null;
	}

	public function getCurrentCartId(): int
	{
		if (BotDetect::isBot()) {
			return 0;
		}

		return $this->getSession()->get('cartId') ?: 0;
	}

	/** Vrati soucasny kosik podle ID ze session. Kdyz kosik neexistuje, vytvori novy
	 */
	public function getCurrentCart(): ?Dao\Cart
	{
		if (BotDetect::isBot()) {
			return new Dao\Cart();
		}

		$cartId = $this->getCurrentCartId();

		if (is_null($cartId) || $cartId === 0) {
			return new Dao\Cart();
		} else {
			if (isset($this->cartsCached[$cartId])) {
				return $this->cartsCached[$cartId];
			}

			$cartRaw = $this->loadRawCart($cartId);

			$dateTime = (new DateTime())->modify('-1 hour');
			if ($cartRaw->lastActivity instanceof DateTimeInterface && $cartRaw->lastActivity->modify('1 minute')
					->getTimestamp() <= $dateTime->getTimestamp()) {
				$this->em->getConnection()
					->executeStatement("UPDATE eshop_orders__cart SET `last_activity` = ? WHERE `id` = ?", [
						(new DateTime())->format('Y-m-d H:i:s'),
						$cartId,
					]);
			}
		}

		$cartId                     = $cartRaw->getId();
		$this->cartsCached[$cartId] = $this->fillDao($cartRaw);

		$cart = $this->cartsCached[$cartId] ?? null;

		if ($cart) {
			foreach ($cart->getCartItems() as &$item) {
				if ($item->getProductId()) {
					$q = $this->checkProductQuantity($item->getProductId(), $item->getQuantity());

					if ($q <= 0) {
						$this->removeItem($item->getId());
					} else {
						$item->setQuantity($q);
					}
				}

				foreach ($item->getChilds() as $k => $child) {
					if (!$child->getProductId()) {
						continue;
					}

					$q = $this->checkProductQuantity($child->getProductId(), $child->getQuantity());


					if ($q <= 0) {
						$item->removeChild($k);
					}
				}
			}

			return $cart;
		}

		return null;
	}

	/** smaze kosik pro daneho uzivatele - po dokonceni objednavky
	 * pak, pri volani getCurrentCart se vygeneruje novy
	 */
	public function deleteCurrentCart()
	{
		$oldCartId = $this->getSession()->get('cartId');
		$this->removeRawCart($oldCartId);
		$this->getSession()->remove('cartId');
		unset($this->cartsCached[$oldCartId]);
	}

	public function getCurrentCartItemsPrice(): float
	{
		$id = $this->getCurrentCartId();
		if (!$id) {
			return 0;
		}

		$rawCart = $this->loadRawCart($id);

		return $this->getDaoCart($rawCart)->getCartItemsPrice();
	}

	private function getDaoCart(Cart $rawCart): Dao\Cart
	{
		$cart = new Dao\Cart();
		$cart
			->setId($rawCart->getId())
			->setIdent($rawCart->ident)
			->setCartItems($this->fillDaoItems($rawCart->getCartItems()));
		AppState::setState('eshopOrdersCartDecimals', $this->currencies->getCurrent()->decimals);

		$cart->orderGiftId = $rawCart->orderGiftId;

		if ($this->user->isLoggedIn() && str_starts_with(Module::$currentPresenterName, 'EshopOrders')) {
			$customer = $this->customers->getByUser($this->user->getIdentity());

			if ($customer && $customer->hasMinimalOrderPrice() > 0) {
				$cart->minimalOrderPrice = $this->exchange->change($customer->hasMinimalOrderPrice());
			}
		}

		return $cart;
	}

	/** nacte kosik z databaze do vlastni property
	 *
	 * @param int $cartId
	 */
	private function loadRawCart($cartId): Cart
	{
		if (!array_key_exists($cartId, $this->cRawCart)) {
			$cartRawQuery = $this->getEr()->createQueryBuilder('c', 'c.id')
				->addSelect('ci')
				->leftJoin('c.cartItems', 'ci')
				->andWhere('c.id = :id')
				->setParameter('id', $cartId);

			$rawCart = $cartRawQuery->getQuery()->getOneOrNullResult();

			if (is_null($rawCart)) {
				$rawCart = $this->createRawCart();

				$this->getSession()->set('cartId', $rawCart->getId());
			}

			$this->cRawCart[$cartId] = $rawCart;
		}

		return $this->cRawCart[$cartId];
	}

	private function createRawCart(): ?Cart
	{
		if (BotDetect::isBot()) {
			return null;
		} else {
			$ident = substr(md5((string) random_int(0, mt_getrandmax())), 0, 7);
			$cart  = new Cart($ident);
			$this->em->persist($cart);
			$this->em->flush($cart);
		}

		return $cart;
	}

	/**
	 * @param $cartId id kosiku ke smazani
	 */
	private function removeRawCart($cartId)
	{
		/** @var Cart $cartRaw */
		$cartRaw = $this->getEr()->find($cartId);

		foreach ($cartRaw->getCartItems() as $item) {
			FileSystem::delete(UPLOADS_DIR . DS . 'eshoporders-cart-uploads' . DS . $item->getId());
		}

		$this->em->remove($cartRaw);
		$this->em->flush();
	}

	/** nacte jednu polozku kosiku z databaze
	 *
	 * @param int $itemId
	 *
	 * @return CartItem
	 */
	private function loadRawCartItem($itemId)
	{
		$cartRawQuery = $this->em->getRepository(CartItem::class)->createQueryBuilder('ci', 'ci.id');
		$cartRawQuery->andWhere('ci.id = :item_id')->setParameter('item_id', $itemId);

		$cartItemRaw = $cartRawQuery->getQuery()->setMaxResults(1)->getOneOrNullResult();

		return $cartItemRaw;
	}

	/**
	 * Return
	 * 0 = chyba
	 * 1 = OK
	 * 2 = Dosahnuto maximalniho mnozstvi
	 */
	public function addItem(AddedCartItem $item): int
	{
		if (BotDetect::isBot()) {
			return 0;
		}

		$productId = $item->productId ? (int) $item->productId : null;
		if ($productId) {
			$quantity  = !$item->ignoreValidation ? $this->checkProductQuantity(
				$productId,
				(int) $item->quantity,
			) : (int) $item->quantity;
			$optionId  = $item->optionId;
			$itemIdent = $productId . '|' . $optionId; //pripadne dalsi vlastnosti s vlivem na cenu
		} else {
			$itemIdent = $item->itemId;
			$quantity  = $item->quantity;
		}

		if ($item->parentId) {
			$itemIdent = 'P' . $item->parentId . '|' . $itemIdent;
		}

		if (isset($item->moreData['disableStacking']) && $item->moreData['disableStacking'] === true) {
			$itemIdent .= '|tm' . time();
		}

		$cartId = $this->getCurrentCartId();

		$cartRaw = $this->loadRawCart($cartId);

		$itemRaw     = $cartRaw->getCartItems()->get($itemIdent);
		$isInRawCart = $itemRaw ? true : false;

		if ($itemRaw) {
			//kdyz polozka uz je v kosiku, jen zvysime pocet
			$itemRaw->quantity = $itemRaw->quantity + $quantity;
			$q1                = $itemRaw->quantity;
		} else {
			//kdyz zatim neexistuje, pridame ji do kosiku
			$itemRaw = new CartItem($itemIdent, $productId ?? null, $cartRaw, $quantity);
			$q1      = $itemRaw->quantity;

			if ($productId) {
				$product = $this->productsFacadeService->getProduct($productId);
				if ($product->package) {
					foreach ($product->package->items as $packageItem) {
						$productItem = $packageItem->product;
						$childRaw    = new CartItem(
							$productItem->id . '|',
							$productItem->id,
							$cartRaw,
							$packageItem->quantity * $itemRaw->quantity
						);
						$childRaw->setParent($itemRaw);

						$this->em->persist($childRaw);
					}
				}
			}
		}

		$itemRaw->quantity = (!$item->ignoreValidation && $productId)
			? $this->checkProductQuantity((int) $itemRaw->getProductId(), (int) $itemRaw->quantity)
			: (int) $item->quantity;
		$q2                = $itemRaw->quantity;

		$itemRaw->setMoreData($item->moreData);
		if ($item->parentId) {
			$itemRaw->setParent($this->em->getReference(CartItem::class, $item->parentId));
		}

		$itemRaw->setMoreDataValue('extraPrice', $item->extraPrice);

		$itemRaw->name  = $item->name;
		$itemRaw->image = $item->image;
		$itemRaw->price = $item->price;

		$this->em->persist($itemRaw);
		$this->em->flush();

		$item->itemId = $itemRaw->getId();

		if (!$isInRawCart) {
			$cartRaw->cartItems->set($item->itemId, $itemRaw);
		}

		//vymazeme kosik z cache, aby se priste obnovil z DB.
		//Alternativni moznost - vzit soucasne DAO a pridat mu polozku
		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);

		if ($q1 > $q2) {
			Flashes::addFlashMessage('eshopOrdersFront.cart.maximumProductQuantityInCart', Flashes::FLASH_INFO);

			return 2;
		}

		return 1;
	}

	/**
	 * Return
	 * 0 = Smazano
	 * 1 = Aktualizovano
	 * 2 = Dosahnuto maximalniho mnozstvi
	 */
	public function updateItemQuantity($itemId, $quantity, array $moreData = []): int
	{
		if ($quantity <= 0 || !$itemId) {
			$this->removeItem($itemId);

			return 0;
		}

		$cartId  = $this->getCurrentCartId();
		$itemRaw = $this->loadRawCartItem($itemId);

		if (!$itemRaw) {
			$this->removeItem($itemId);

			return 0;
		}

		$q1                = $itemRaw->quantity;
		$itemRaw->quantity = $this->checkProductQuantity((int) $itemRaw->getProductId(), (int) $quantity);
		$q2                = $itemRaw->quantity;

		if (isset($moreData['note'])) {
			$itemRaw->setMoreDataValue('note', $moreData['note']);
		}

		$product = $this->productsFacadeService->getProduct($itemRaw->getProductId());
		foreach ($itemRaw->getChildren() as $child) {
			if ($product->package) {
				$packageItem = $product->package->items[$child->getProductId()] ?? null;
				if ($packageItem) {
					$child->quantity = $packageItem->quantity * $itemRaw->quantity;
					$this->em->persist($child);
				}
			}
		}

		$this->em->persist($itemRaw);
		$this->em->flush();

		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);

		if ($q1 > $q2 && $quantity > $q1) {
			Flashes::addFlashMessage('eshopOrdersFront.cart.maximumProductQuantityInCart', Flashes::FLASH_INFO);

			return 2;
		}

		return 1;
	}

	/**
	 * Return
	 * 0 = chyba
	 * 1 = Smazano
	 */
	public function removeItem($itemId): int
	{
		try {
			$cartId = $this->getCurrentCartId();

			$itemRaw = $this->loadRawCartItem($itemId);
			if ($itemRaw) {
				$this->em->remove($itemRaw);
				$this->em->flush();
			}

			FileSystem::delete(UPLOADS_DIR . DS . 'eshoporders-cart-uploads' . DS . $itemId);

			$this->cDaoItems = null;
			unset($this->cartsCached[$cartId]); //obnoveni cache
		} catch (Exception $e) {
			Debugger::log($e);

			return 0;
		}

		return 1;
	}

	public function setOrderGift(?int $giftId = null)
	{
		$cartId = $this->getCurrentCartId();
		if (!$cartId)
			return;

		$cartRaw = $this->loadRawCart($cartId);

		$cartRaw->orderGiftId = $giftId;
		$this->em->persist($cartRaw);
		$this->em->flush();

		$cart              = $this->getCurrentCart();
		$cart->orderGiftId = $giftId;
	}

	public function clearCartItems()
	{
		$cartId = $this->getCurrentCartId();
		if (!$cartId)
			return;

		$cart = $this->loadRawCart($cartId);

		foreach ($cart->cartItems->toArray() as $item) {
			$this->em->remove($item);
		}

		$this->em->flush();
		$this->cDaoItems = null;
		unset($this->cartsCached[$cartId]);
	}

	protected function checkProductQuantity(int $productId, int $quantity): int
	{
		$product = $this->productsFacadeService->getProduct($productId);

		if (!$product) {
			return $quantity;
		}

		if (!$product->canAddToCart || !$product->getAvailability() || !$product->getAvailability()->canAddToCart()) {
			return 0;
		}

		$cartProductMaxQuantityByEshop = Config::load('cartProductMaxQuantityByEshop', []) ?? [];

		if (
			(Arrays::contains($cartProductMaxQuantityByEshop, $this->sites->getCurrentSite()->getIdent())
				&& $quantity > $product->getQuantity() + $product->getQuantityExternal()
				&& (!$product->availabilityAfterSoldOut || !$product->getAvailability()->canAddToCart()))
			|| (EshopCatalogConfig::load('pseudoWarehouse')
				&& (int) $product->unlimitedQuantity === 0
				&& (!EshopCatalogConfig::load('allowWarehouseOverdraft') || !$product->getAvailability()->warehouseOverdraft)
				&& $quantity > $product->getQuantity() + $product->getQuantityExternal())
		) {
			$quantity = $product->getQuantity() + $product->getQuantityExternal();
		}

		if ($product->getMaximumAmount() !== null && $quantity > $product->getMaximumAmount()) {
			$quantity = (int) $product->getMaximumAmount();
		}

		return $quantity;
	}

	/**
	 * @param Cart $cartRaw
	 *
	 * @return Dao\Cart
	 */
	protected function fillDao($cartRaw)
	{
		$cart = $this->getDaoCart($cartRaw);

		if ($cart->getCartItems()) {
			$this->eventDispatcher->dispatch(new FillDaoEvent($cart, $cartRaw), 'eshopOrders.cartFillDao');
		}

		return $cart;
	}

	/**
	 * @param Entity\CartItem[] $cartItemsRaw
	 *
	 * @return ArrayCollection(Dao\CartItem) polozky
	 */
	protected function fillDaoItems($cartItemsRaw)
	{
		if ($cartItemsRaw instanceof ArrayCollection || $cartItemsRaw instanceof PersistentCollection) {
			$cartItemsRaw = $cartItemsRaw->toArray();
		}

		$this->eventDispatcher->dispatch(new FillDaoItemsEvent($cartItemsRaw), 'eshopOrders.cartFillDaoItems');

		return $this->cDaoItems;
	}

	protected function getSession(): SessionSection
	{
		if ($this->sessionSection === null) {
			$this->sessionSection = $this->session->getSection('eshopOrdersCart');
			$this->sessionSection->setExpiration('1 month');
		}

		return $this->sessionSection;
	}
}

