<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\NonUniqueResultException;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\EshopOrdersConfig;
use Users\Model\Entities\User;
use Users\Model\Roles;
use Users\Model\Users;

/**
 * class Customers
 * @package EshopOrders\FrontModule\Model
 *
 * @method Customer|object|null getReference($id)
 * @method Customer[]|null getAll()
 * @method Customer|null get($id)
 */
class Customers extends BaseEntityService
{
	final public const CUSTOMER = 'customer';

	protected $entityClass = Customer::class;

	protected array $cCustomers = [];

	public function __construct(protected Users $usersService, protected Roles $rolesService)
	{
	}

	/**
	 * @throws NonUniqueResultException
	 */
	public function getByUser(User|int $user): ?Customer
	{
		if ($user instanceof User) {
			$userId = $user->getId();
		} else {
			$userId = (int) $user;
		}

		if (!array_key_exists($userId, $this->cCustomers)) {
			$qb = $this->getEr()->createQueryBuilder('c', 'c.id')
				->addSelect('addrAv')
				->leftJoin('c.addressesAvailable', 'addrAv')
				->andWhere('c.user = :user')->setParameter('user', $userId);

			if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer', false)) {
				$qb->leftJoin('c.disabledPayments', 'disP')
					->leftJoin('c.addressDelivery', 'deli')
					->leftJoin('c.addressInvoice', 'inv')
					->leftJoin('c.disabledSpeditions', 'disS')
					->leftJoin('c.groupCustomers', 'gc')
					->leftJoin('gc.params', 'gcp')
					->addSelect('disP, disS, deli, inv, gc, gcp');
			}

			$this->cCustomers[$userId] = $qb->getQuery()->getOneOrNullResult();
		}

		return $this->cCustomers[$userId] ?? null;
	}

	/**
	 * @param int $userId
	 *
	 * @throws NonUniqueResultException
	 */
	public function getByUserId(?int $userId): ?Customer
	{
		return $userId ? $this->getByUser($userId) : null;
	}

	public function getOrCreateCustomer($userId, $email = null, $firstName = '', $lastName = '', $phone = null)
	{
		//user
		$user = $this->usersService->getOrCreateUser($userId, $email, $firstName, $lastName);

		//ma user prava "customer"? Pokud ne, tak pridat prava
		if (!$user->isInRoleByIdent(self::CUSTOMER)) {
			$role = $this->rolesService->getByIdent(self::CUSTOMER);
			if ($role) {
				$user->addRole($role);
				$this->em->persist($user)->flush();
			}
		}

		//customer
		$customer = $this->getByUser($user);
		if (!$customer) {
			$customer = new Customer($user, $phone);
			$this->em->persist($customer);
			$this->em->flush();

			$this->cCustomers = [];
			$customer         = $this->getByUser($user);
		}

		return $customer;
	}
}
