<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Components\Customer\AddressesList;
use EshopOrders\FrontModule\Components\Customer\IAddressesListFactory;
use EshopOrders\FrontModule\Components\Customer\IOrdersListFactory;
use EshopOrders\FrontModule\Components\Customer\IProfileFormFactory;
use EshopOrders\FrontModule\Components\Customer\ProfileForm;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Orders;
use EshopOrders\FrontModule\Model\ProductsAssignedToCustomers;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Nette\Application\BadRequestException;
use Nette\Application\UI\Multiplier;
use Users\Model\Users;

class CustomerPresenter extends BasePresenter
{
	public function __construct(
		protected Orders                      $ordersService,
		protected Users                       $usersService,
		protected Customers                   $customersService,
		public Invoices                       $invoices,
		public IInvoicePdfBuilderFactory      $invoicePdfBuilderFactory,
		protected ProductsAssignedToCustomers $productsAssignedToCustomers,
		protected ProductsFacade              $productsFacade,
	)
	{
	}

	public function checkRequirements($element): void
	{
		parent::checkRequirements($element);
		if (!$this->getUser()->isLoggedIn()) {
			$this->redirect(':Users:Front:Login:');
		}
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault()
	{
		$this->template->messages      = $this->getTemplate()->flashes;
		$this->template->hasMyProducts = (bool) $this->productsAssignedToCustomers->getProductsIdByUser(
			$this->getUser()
				->getId(),
		);
		$this->getTemplate()->flashes  = [];
	}

	public function actionOrderDetail(int $orderId)
	{
		$customer = $this->customersService->getByUser($this->getUser()->getIdentity());
		$order    = $this->ordersService->getDetail($orderId);

		if (
			!$order->getCustomer()
			|| $order->getCustomer()->getId() != $customer->getId()
			|| $order->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent()
		)
			throw new BadRequestException();

		$tmp = [];
		foreach ($order->getOrderItems() as $item) {
			if ($item->getProductId()) {
				$tmp[] = $item->getProductId();
			}
		}

		$tmp                      = array_unique($tmp);
		$this->template->products = !empty($tmp) ? $this->productsFacade->getProducts($tmp) : [];
		$this->template->order    = $order;
	}

	public function actionMyProducts()
	{
		if (!Config::load('product.allowAssignProductToCustomer')) {
			$this->error();
		}

		$ids = $this->productsAssignedToCustomers->getProductsIdByUser($this->getUser()->getId());

		if (!empty($ids)) {
			$this->template->products = $this->productsFacade->getProducts($ids);
		} else {
			$this->template->products = [];
		}
	}

	public function actionEditProfile()
	{
	}

	public function actionInvoice(int $id)
	{
		if (!EshopOrdersConfig::load('invoice.enable', false))
			throw new BadRequestException();

		/** @var Order $order */
		$order    = $this->ordersService->getDetail($id);
		$customer = $this->customersService->getByUser($this->getUser()->getIdentity());

		if (!$order || !$order->getCustomer() || !$customer || $order->getCustomer()->getId() != $customer->getId())
			throw new BadRequestException();

		$invoice = $this->invoices->getOneByOrder($order, false);

		if (!$invoice)
			throw new BadRequestException();

		$this->invoicePdfBuilderFactory->create($invoice)->render();
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersList(IOrdersListFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentProfileForm(IProfileFormFactory $factory): ProfileForm
	{
		$control = $factory->create($this->getUser()->getIdentity());

		$control['form']->onSuccess['redirect'] = function(): never {
			$this->redirect('this');
		};

		return $control;
	}

	protected function createComponentAddressesList(IAddressesListFactory $factory): AddressesList
	{
		$control = $factory->create($this->getUser()->getId());

		$control->onAddressSave[] = function(AddressesList $control) {
			$this['profileForm']->redrawControl('formAddresses');
			$this['profileForm']->redrawControl('formAddresses2');
		};

		return $control;
	}

	protected function createComponentProductPreview(IProductPreviewFactory $factory): Multiplier
	{
		return new Multiplier(static function($id) use ($factory) {
			$control = $factory->create();
			$control->setProductById((int) $id);

			return $control;
		});
	}
}
