<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use Users\Model\Entities\User;

#[ORM\Table('eshop_orders__customer')]
#[ORM\Entity]
#[ORM\EntityListeners(['EshopOrders\Model\Listeners\UserListener'])]
class Customer
{
	use TId;
	use TExtraField;

	const EXTRA_FIELD_SECTION = 'eshopOrdersCustomer';

	#[ORM\OneToOne(targetEntity: 'Users\Model\Entities\User')]
	#[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public User $user;

	#[ORM\Column(name: 'phone', type: 'string', nullable: true)]
	protected ?string $phone = null;

	/** defaultni (prave vybrana) dorucovaci adresa
	 */
	#[ORM\ManyToOne(targetEntity: 'CustomerAddress')]
	#[ORM\JoinColumn(name: 'address_delivery', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	public ?CustomerAddress $addressDelivery = null;

	/** defaultni (prave vybrana) dorucovaci adresa
	 */
	#[ORM\ManyToOne(targetEntity: 'CustomerAddress')]
	#[ORM\JoinColumn(name: 'address_invoice', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	public ?CustomerAddress $addressInvoice = null;

	/**
	 * @var Collection<CustomerAddress>
	 */
	#[ORM\JoinTable(name: 'eshop_orders__customer_customer_address')]
	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'address_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: 'CustomerAddress', indexBy: 'id', inversedBy: 'customers')]
	public Collection $addressesAvailable;

	/**
	 * @var Collection<Order>
	 */
	#[ORM\OneToMany(targetEntity: 'Order', mappedBy: 'customer', indexBy: 'id')]
	public Collection $orders;

	#[ORM\ManyToOne(targetEntity: 'GroupCustomers', cascade: ['persist'])]
	#[ORM\JoinColumn(name: 'id_group_customers', referencedColumnName: 'id', onDelete: 'SET NULL', nullable: true)]
	protected ?GroupCustomers $groupCustomers;

	/**
	 * @var Payment[]|ArrayCollection
	 */
	#[ORM\JoinTable(name: 'eshop_orders__customer_disabled_payment')]
	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'payment_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: 'Payment', indexBy: 'id')]
	public $disabledPayments;

	/**
	 * @var Spedition[]|ArrayCollection
	 */
	#[ORM\JoinTable(name: 'eshop_orders__customer_disabled_spedition')]
	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'spedition_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: 'Spedition', indexBy: 'id')]
	public $disabledSpeditions;

	public function __construct(
		User    $user,
		?string $phone,
	)
	{
		$this->user               = $user;
		$this->phone              = $phone;
		$this->addressesAvailable = new ArrayCollection();
		$this->orders             = new ArrayCollection();
		$this->disabledPayments   = new ArrayCollection();
		$this->disabledSpeditions = new ArrayCollection();
	}

	/**
	 * @return User
	 */
	public function getUser(): User
	{
		return $this->user;
	}

	/**
	 * @return string
	 */
	public function getPhone(): string
	{
		return $this->phone ?: "";
	}

	/**
	 * @param string $phone
	 *
	 * @return Customer
	 */
	public function setPhone(string $phone): Customer
	{
		$this->phone = $phone;

		return $this;
	}

	/**
	 * @return CustomerAddress
	 */
	public function getAddressDelivery()
	{
		return $this->addressDelivery;
	}

	public function setAddressDelivery(?CustomerAddress $addressDelivery): Customer
	{
		$this->addressDelivery = $addressDelivery;
		if ($addressDelivery && !$this->addressesAvailable->contains($addressDelivery)) {
			$this->addressesAvailable->add($addressDelivery);
		}

		return $this;
	}

	/**
	 * @return CustomerAddress
	 */
	public function getAddressInvoice(): ?CustomerAddress
	{
		return $this->addressInvoice;
	}

	public function setAddressInvoice(?CustomerAddress $addressInvoice): Customer
	{
		$this->addressInvoice = $addressInvoice;
		if ($addressInvoice && !$this->addressesAvailable->contains($addressInvoice)) {
			$this->addressesAvailable->add($addressInvoice);
		}

		return $this;
	}

	/**
	 * @return ArrayCollection|Order[]
	 */
	public function getOrders()
	{
		return $this->orders;
	}

	/** TODO pridavat a ubirat po jedne?
	 *
	 * @param ArrayCollection|Order[] $orders
	 *
	 * @return Customer
	 */
	public function setOrders($orders)
	{
		$this->orders = $orders;

		return $this;
	}

	public function getGroupCustomers(): ?GroupCustomers
	{
		return $this->groupCustomers;
	}

	public function setGroupCustomers(GroupCustomers $groupCustomers)
	{
		$this->groupCustomers = $groupCustomers;
	}

	public function removeGroupCustomers()
	{
		$this->groupCustomers = null;
	}

	public function hasMinimalOrderPrice(): float
	{
		if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
			$ef = $this->getExtraFieldsValues();

			if (isset($ef['allowMinimalOrderPrice']) && $ef['allowMinimalOrderPrice'] == true && $ef['minimalOrderPrice'] > 0) {
				return (float) str_replace(',', '.', (string) $ef['minimalOrderPrice']);
			}
		}

		return 0;
	}
}

