<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;

#[ORM\Table('eshop_orders__customer_address')]
#[ORM\Entity]
#[ORM\EntityListeners(['EshopOrders\Model\Listeners\UserListener'])]
class CustomerAddress
{
	use TId;

	#[ORM\Column(name: 'address_name', type: 'string', length: 255, nullable: true)]
	public ?string $addressName = null;

	#[ORM\Column(name: 'first_name', type: 'string', nullable: true)]
	public ?string $firstName = null;

	#[ORM\Column(name: 'last_name', type: 'string', nullable: true)]
	public ?string $lastName = null;

	#[ORM\Column(name: 'email', type: 'string', length: 255, nullable: true)]
	public ?string $email = null;

	#[ORM\Column(name: 'phone', type: 'string', length: 255, nullable: true)]
	public ?string $phone = null;

	#[ORM\Column(name: 'company', type: 'string', nullable: true)]
	public ?string $company = null;

	#[ORM\Column(name: 'street', type: 'string', nullable: true)]
	public ?string $street = null;

	#[ORM\Column(name: 'city', type: 'string', nullable: true)]
	public ?string $city = null;

	#[ORM\Column(name: 'postal', type: 'string', nullable: true)]
	public ?string $postal = null;

	#[ORM\Column(name: 'id_number', type: 'string', nullable: true)]
	public ?string $idNumber = null;

	#[ORM\Column(name: 'vat_number', type: 'string', nullable: true)]
	public ?string $vatNumber = null;

	#[ORM\Column(name: 'id_vat_number', type: 'string', length: 255, nullable: true)]
	public ?string $idVatNumber = null;

	#[ORM\Column(name: 'customer_id', type: 'integer', options: ['unsigned' => true], nullable: true)]
	protected ?int $customer = null;

	#[ORM\ManyToOne(targetEntity: 'Core\Model\Entities\Country')]
	#[ORM\JoinColumn(name: 'country_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
	public ?Country $country = null;

	#[ORM\Column(name: 'validated_id_number', type: 'smallint', length: 1, options: ['default' => 0])]
	public int $validatedIdNumber = 0;

	#[ORM\Column(name: 'validated_vat_number', type: 'smallint', length: 1, options: ['default' => 0])]
	public int $validatedVatNumber = 0;

	/**
	 * @var Collection<Customer>
	 */
	#[ORM\ManyToMany(targetEntity: 'Customer', mappedBy: 'addressesAvailable', indexBy: 'id')]
	public Collection $customers;

	public function __construct(Customer $customer)
	{
		$customer->addressesAvailable->add($this);
		$this->customers = new ArrayCollection();
	}

	public function getFirstName(): ?string { return $this->firstName; }

	public function setFirstName(?string $firstName): self
	{
		$this->firstName = $firstName;

		return $this;
	}

	public function getLastName(): ?string { return $this->lastName; }

	public function setLastName(?string $lastName): self
	{
		$this->lastName = $lastName;

		return $this;
	}

	public function getName(): string { return trim($this->getFirstName() . ' ' . $this->getLastName()); }

	public function getCompany(): ?string { return $this->company; }

	public function setCompany(?string $company): self
	{
		$this->company = $company;

		return $this;
	}

	/*******
	 * === Street
	 */

	public function getStreet()
	{
		return $this->street;
	}

	public function setStreet($street)
	{
		$this->street = $street;

		return $this;
	}

	/*******
	 * === City
	 */

	public function getCity()
	{
		return $this->city;
	}

	public function setCity($city)
	{
		$this->city = $city;

		return $this;
	}

	/*******
	 * === Postal
	 */

	public function getPostal()
	{
		return $this->postal;
	}

	public function setPostal($postal)
	{
		$this->postal = $postal;

		return $this;
	}

	/*******
	 * === IdNumber - ICO
	 */

	public function getIdNumber()
	{
		return $this->idNumber;
	}

	public function setIdNumber($idNumber): self
	{
		$this->idNumber = $idNumber;

		return $this;
	}

	/*******
	 * === VatNumber - DIC
	 */

	public function getVatNumber()
	{
		return $this->vatNumber;
	}

	public function setVatNumber($vatNumber): self
	{
		$this->vatNumber = $vatNumber;

		return $this;
	}

	public function getCountry(): ?Country { return $this->country; }

	public function setCountry(Country $country): self
	{
		$this->country = $country;

		return $this;
	}

	public function toArray(): array
	{
		return [
			'firstName' => $this->getFirstName(),
			'lastName'  => $this->getLastName(),
			'company'   => $this->company,
			'email'     => $this->email,
			'phone'     => $this->phone,
			'street'    => $this->getStreet(),
			'city'      => $this->getCity(),
			'postal'    => $this->getPostal(),
			'country'   => $this->country ? $this->country->getId() : null,
			'idNumber'  => $this->idNumber,
			'vatNumber' => $this->vatNumber,
		];
	}

	public function fillFromOrderAddress(OrderAddress $address): self
	{
		$this->setFirstName($address->getFirstName());
		$this->setLastName($address->getLastName());
		$this->email       = $address->getEmail();
		$this->phone       = $address->getPhone();
		$this->street      = $address->getStreet();
		$this->city        = $address->getCity();
		$this->postal      = $address->getPostal();
		$this->country     = $address->getCountry();
		$this->idVatNumber = $address->idVatNumber;
		$this->setCompany($address->getCompany());

		if (EshopOrdersConfig::load('customer.disableModifyIco')) {
			if (!$this->idNumber) {
				$this->idNumber = $address->getIdNumber();
			}

			if (!$this->vatNumber) {
				$this->vatNumber = $address->getVatNumber();
			}
		} else {
			$this->idNumber  = $address->getIdNumber();
			$this->vatNumber = $address->getVatNumber();
		}

		return $this;
	}

	public function getDetailString(): string
	{
		$str = '';

		if ($this->addressName) {
			$str .= $this->addressName . ' - ';
		}

		$str .= trim(implode(', ', [$this->getName(), $this->getStreet(), $this->getPostal(), $this->getCity()]), ' ,');

		return $str;
	}
}
