<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice\InvoiceData;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\DateTime;

#[ORM\Table('eshop_orders__invoice')]
#[ORM\Entity]
#[ORM\EntityListeners(['EshopOrders\Model\Listeners\InvoiceListener'])]
class Invoice
{
	use TId;

	/**
	 * @var DateTime|null
	 */
	#[ORM\Column(type: 'datetime', nullable: true)]
	public $dueDate;

	/**
	 * @var string
	 */
	#[ORM\Column(type: 'string', nullable: false)]
	public $ident;

	/**
	 * @var Order
	 */
	#[ORM\OneToOne(targetEntity: 'Order', mappedBy: 'invoice', cascade: ['persist'])]
	public $order;

	/**
	 * @var DateTime
	 */
	#[Gedmo\Timestampable(on: 'create')]
	#[ORM\Column(type: 'datetime', nullable: false)]
	public $createdAt;

	/**
	 * @var InvoiceData|null
	 */
	#[ORM\OneToOne(targetEntity: 'EshopOrders\Model\Entities\Invoice\InvoiceData', mappedBy: 'invoice', cascade: ['persist'])]
	public $invoiceData;

	/**
	 * @var Seller
	 */
	#[ORM\ManyToOne(targetEntity: 'EshopCatalog\Model\Entities\Seller')]
	#[ORM\JoinColumn(name: 'seller_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public $seller;

	#[ORM\Column(type: 'string', nullable: true)]
	protected ?string $externalInvoiceFile = null;

	#[ORM\Column(type: 'text', nullable: true, options: ['default' => null])]
	public ?string $note = null;

	public function __construct(?DateTime $dueDate, string $ident, Seller $seller)
	{
		$this->dueDate = $dueDate;
		$this->ident   = $ident;
		$this->seller  = $seller;
	}

	public function setExternalInvoiceFile(?string $externalInvoiceFile): void
	{
		if ($externalInvoiceFile === null) {
			$this->externalInvoiceFile = null;

			return;
		}

		if (file_exists(WWW_DIR . $externalInvoiceFile)) {
			$this->externalInvoiceFile = $externalInvoiceFile;
		}
	}

	public function getExternalInvoiceFile(): ?string
	{
		if ($this->externalInvoiceFile !== null && file_exists(WWW_DIR . $this->externalInvoiceFile)) {
			return $this->externalInvoiceFile;
		}

		return null;
	}

	public function isExternal(): bool
	{
		return $this->externalInvoiceFile !== null;
	}

}
