<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Currency\Model\Config;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;

#[ORM\Table('eshop_orders__order_discount')]
#[ORM\Entity]
class OrderDiscount implements IDiscount
{
	const TYPE_PERCENT        = 'percent';
	const TYPE_FIX            = 'fix';
	const TYPE_DELIVERY_PRICE = 'deliveryPrice';

	use TId;

	/**
	 * @var string
	 */
	#[ORM\Column(name: 'code', type: 'string')]
	protected $code;

	//protected $discount_id //odkaz do tabulky se seznamem pouzitelnych slevovych kodu - pridat pozdeji
	/**
	 * @var string
	 */
	#[ORM\Column(name: 'name', type: 'string')]
	protected $name;

	/** sleva v procentech nebo pevna castka
	 * @var int
	 */
	#[ORM\Column(name: 'value', type: 'decimal', precision: 10, scale: 2)]
	protected $value;

	/** vysledna sleva vypocitana z procent - zaporna hodnota
	 * @var float
	 */
	#[ORM\Column(name: 'price', type: 'decimal', precision: 10, scale: 2)]
	protected $price;

	/** procenta / fixni castka
	 * @var string
	 */
	#[ORM\Column(name: 'type', type: 'string')]
	protected $type;

	/**
	 * @var Order
	 */
	#[ORM\ManyToOne(targetEntity: 'Order', inversedBy: 'orderDiscounts')]
	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public $order;

	public function __construct($code, $order)
	{
		$this->code  = $code;
		$this->order = $order;
	}

	public function calculateDiscount(float $price, bool $useCurrency = false): float
	{
		return -abs($this->getPrice($useCurrency));
	}

	/*****
	 * === Code
	 */

	public function getCodeFormatted(): string
	{
		if (in_array((string) $this->code, ['ESHOPSALESAUTO', 'CHECKOUTSALE']))
			return '';

		return (string) $this->code;
	}

	public function getCode()
	{
		return $this->code;
	}

	public function setCode($code): OrderDiscount
	{
		$this->code = $code;

		return $this;
	}

	/*****
	 * === Name
	 */

	public function getName()
	{
		return $this->name;
	}

	public function setName($name): OrderDiscount
	{
		$this->name = $name;

		return $this;
	}

	/*****
	 * === Quantity
	 */

	public function setQuantity($quantity)
	{
		$this->quantity = $quantity && $quantity > 0 ? $quantity : null;
	}

	public function getQuantity()
	{
		return $this->quantity;
	}

	/*******
	 * === Value
	 */

	public function getValue(): float
	{
		return (float) $this->value;
	}

	public function setValue(float $value)
	{
		$this->value = $value;

		return $this;
	}

	/*******
	 * === Price
	 */

	public function getPrice(bool $useCurrency = false): float
	{
		$price = (float) $this->price;

		$price = $useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice($price)
			: $price;

		if (Config::load('roundOrderSales')) {
			$price = round($price);
		}

		return $price;
	}

	public function setPrice(float $price)
	{
		$this->price = Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$price = $this->getPrice($useCurrency);

		return round(OrderHelper::removeVat($price, $this->getVatRate()), 2);
	}

	public function getVat(bool $useCurrency = false)
	{
		return $this->getPrice() - $this->getPriceWithoutVat();
	}

	public function getVatRate(): int
	{
		return (int) EshopOrdersConfig::load('discounts.vat');
	}

	/*******
	 * === Type
	 */

	public function getType(): string
	{
		return $this->type;
	}

	public function setType($type): OrderDiscount
	{
		$this->type = $type;

		return $this;
	}

	/*******
	 * === Order
	 */

	public function getOrder(): Order
	{
		return $this->order;
	}

	public function setOrder(Order $order)
	{
		$this->order = $order;

		return $this;
	}

	public function getSortPosition(): int
	{
		switch ($this->type) {
			case self::TYPE_PERCENT:
				return 1;
			case self::TYPE_FIX:
				return 2;
		}

		return 9999;
	}
}

