<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Helpers\OrderItemReviewsHelper;
use EshopOrders\Model\Helpers\ReviewsHelper;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\DateTime;

#[ORM\Table(name: 'eshop_orders__order_item_review')]
#[ORM\Entity]
class OrderItemReview
{
	public const STATUS_NEW = 'new', STATUS_PUBLISHED = 'published', STATUS_NOT_PUBLISHED = 'notPublished', STATUS_TOP = 'top';
	public const                                                                                            STATUSES   = [self::STATUS_NEW,
		self::STATUS_PUBLISHED, self::STATUS_NOT_PUBLISHED, self::STATUS_TOP];

	use TId;

	#[ORM\ManyToOne(targetEntity: 'OrderItem', inversedBy: 'orderItemReview')]
	#[ORM\JoinColumn(referencedColumnName: 'id', nullable: false, onDelete: 'CASCADE')]
	public OrderItem $orderItem;

	/**
	 * @var DateTime
	 */
	#[Gedmo\Timestampable(on: 'create')]
	#[ORM\Column(type: 'datetime', options: ['default' => 'CURRENT_TIMESTAMP'])]
	public $created;

	/**
	 * @var float
	 */
	#[ORM\Column(type: 'decimal', precision: 10, scale: 1, nullable: false)]
	public $rating;

	#[ORM\Column(name: 'description', type: 'text', nullable: true)]
	public ?string $description;

	#[ORM\Column(type: 'string', options: ['default' => 'new'], nullable: false)]
	public string $status = self::STATUS_NEW;

	#[ORM\Column(type: 'string', nullable: true)]
	public ?string $customer = null;

	public function __construct(OrderItem $orderItem, DateTime $created, float $rating, ?string $description)
	{
		$this->orderItem   = $orderItem;
		$this->created     = $created;
		$this->rating      = Strings::formatEntityDecimal($rating);
		$this->description = $description;
	}

	public function getRating(): float
	{
		return (float) Strings::formatEntityDecimal($this->rating);
	}

	/**
	 * @return float[]
	 */
	public function getRatingAsArray(bool $withBlanks = false): array
	{
		return OrderItemReviewsHelper::ratingToArray($this->getRating(), $withBlanks);
	}

	public function getMaxRating(): float
	{
		return EshopOrders\Model\EshopOrdersConfig::load('orderItemReviews.maxRating');
	}

}
