<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Helpers\OrderHelper;

#[ORM\Table('eshop_orders__order_item_sale')]
#[ORM\Entity]
class OrderItemSale
{
	use TId;

	public const TYPE_PERCENT_ONE = 'percentOne';
	public const TYPE_PERCENT_ALL = 'percentAll';
	public const TYPE_FIX_ONE     = 'fixOne';
	public const TYPE_FIX_ALL     = 'fixAll';

	/**
	 * @var string
	 */
	#[ORM\Column(name: 'name', type: 'string', nullable: true)]
	protected $name;

	/**
	 * @var string
	 */
	#[ORM\Column(name: 'description', length: 1000, type: 'string', nullable: true)]
	protected $description;

	/** sleva v procentech nebo pevna castka
	 * @var float
	 */
	#[ORM\Column(name: 'value', nullable: false, type: 'decimal', precision: 10, scale: 2, options: ['default' => 0])]
	protected $value;

	/**
	 * @var string
	 */
	#[ORM\Column(name: 'type', nullable: false)]
	protected $type;

	/**
	 * @var OrderItem
	 */
	#[ORM\ManyToOne(targetEntity: 'OrderItem', inversedBy: 'sales')]
	#[ORM\JoinColumn(name: 'order_item_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public $orderItem;

	public function __construct($value, $orderItem, $type)
	{
		$this->value     = $value;
		$this->orderItem = $orderItem;
		$this->type      = $type;
	}

	/**
	 * @return string
	 */
	public function getName(): string
	{
		return $this->name;
	}

	/**
	 * @param string $name
	 *
	 * @return OrderItemSale
	 */
	public function setName(string $name): OrderItemSale
	{
		$this->name = $name;

		return $this;
	}

	public function getType(): string
	{
		return $this->type;
	}

	public function setType(string $type): self
	{
		$this->type = $type;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getDescription(): string
	{
		return $this->description;
	}

	/**
	 * @param string $description
	 *
	 * @return OrderItemSale
	 */
	public function setDescription(string $description): OrderItemSale
	{
		$this->description = $description;

		return $this;
	}

	/**
	 * @return float
	 */
	public function getValue(): float
	{
		return (float) $this->value;
	}

	public function setValue(float $value): OrderItemSale
	{
		$this->value = $value;

		return $this;
	}

	/**
	 * @return OrderItem
	 */
	public function getOrderItem(): OrderItem
	{
		return $this->orderItem;
	}

	/**
	 * @param OrderItem $orderItem
	 *
	 * @return OrderItemSale
	 */
	public function setOrderItem(OrderItem $orderItem): OrderItemSale
	{
		$this->orderItem = $orderItem;

		return $this;
	}

	/**
	 * @return int|null
	 */
	public function getId(): ?int
	{
		return $this->id;
	}

	public function getSaleValue(bool $useCurrency = false)
	{
		$basePrice = $this->orderItem->getPrice($useCurrency);
		$quantity  = $this->orderItem->getQuantity();

		return OrderHelper::calculateSaleValue($this->type, $this->getValue(), $quantity, $basePrice);
	}

	public function getVatRate(): int
	{
		return $this->orderItem->getVatRate();
	}

	public function getFormatted()
	{
		$price    = $this->orderItem->getOrder()->renderPrice($this->getValue());
		$quantity = $this->orderItem->getQuantity();
		switch ($this->type) {
			case self::TYPE_FIX_ALL:
				return '-' . $price . ' na produkt';
			case self::TYPE_FIX_ONE:
				return '-' . $price . ' na ks';
			case self::TYPE_PERCENT_ALL:
				return '-' . $this->value . '% na produkt';
			case self::TYPE_PERCENT_ONE:
				return '-' . $this->value . '% na ks';
		}
	}

}
