<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Contributte\Translation\Exceptions\InvalidArgument;
use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Event\EventDispatcher;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Doctrine;
use Doctrine\ORM\Event\PostUpdateEventArgs;
use Doctrine\ORM\Event\PrePersistEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Event\OrderStatusEvent;
use Exception;
use Nette\SmartObject;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderListener implements EventSubscriberInterface
{
	use SmartObject;

	public function __construct(
		protected PriceFilter     $priceFilter,
		protected EventDispatcher $eventDispatcher,
		protected Translator      $translator,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	#[ORM\PrePersist]
	public function prePersistHandler(Order $order, PrePersistEventArgs $event): void
	{
		$em = $event->getObjectManager();
		if ($order->isPaid === 1 && $order->getId() === null) {
			$em->persist($this->createOrderPaidHistory($order, true));
		}
	}

	#[ORM\PostUpdate]
	public function postUpdateHandler(Order $order, PostUpdateEventArgs $event): void
	{
		$em        = $event->getObjectManager();
		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($order);

		if ($changeSet && isset($changeSet['isPaid'])) {
			$isPaid           = $changeSet['isPaid'][1] === 1;
			$orderPaidHistory = $this->createOrderPaidHistory($order, $isPaid);
			$em->persist($orderPaidHistory);

			if ($isPaid) {
				$order->paid = new DateTime;
				$this->eventDispatcher->dispatch(
					new OrderStatusEvent($order, OrderStatus::STATUS_IS_PAID),
					Order::class . '::changeStatus',
				);
			} else {
				try {
					$em->flush($orderPaidHistory);
				} catch (Exception $e) {
					Debugger::log($e, 'order-listener-log-paid');
				}
			}
		}
	}

	#[ORM\PostLoad]
	public function postLoad(Order $order, Doctrine\ORM\Event\PostLoadEventArgs $args): void
	{
		$order->priceFilter = clone $this->priceFilter;

		$this->eventDispatcher->dispatch(new OrderEvent($order), Order::class . '::postLoad');
	}

	/**
	 * @param Order $order
	 * @param bool  $isPaid
	 *
	 * @return OrderPaidHistory
	 * @throws InvalidArgument
	 */
	protected function createOrderPaidHistory(Order $order, bool $isPaid): OrderPaidHistory
	{
		$eshopOrdersPaymentMethodText = $this->translator->translate(
			AppState::getState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.default'),
		);
		$orderPaidHistory             = new OrderPaidHistory($order, $eshopOrdersPaymentMethodText);
		$orderPaidHistory->isDeleted  = (int) !$isPaid;

		return $orderPaidHistory;
	}

}
