<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Helpers\OrderHelper;

/** podobne jako order_item, ale samostatna tabulka
 */
#[ORM\Table('eshop_orders__order_spedition')]
#[ORM\Entity]
class OrderSpedition
{
	use TId;

	/**
	 * @var Spedition
	 */
	#[ORM\ManyToOne(targetEntity: 'Spedition')]
	#[ORM\JoinColumn(name: 'spedition_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
	public $spedition;

	/**
	 * @var string
	 */
	#[ORM\Column(name: 'name', type: 'string')]
	protected $name;

	/**
	 * @var float
	 */
	#[ORM\Column(name: 'price', type: 'decimal', precision: 10, scale: 2)]
	protected $price;

	/**
	 * @var Order
	 */
	#[ORM\OneToOne(targetEntity: 'Order', inversedBy: 'spedition')]
	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public $order;

	#[ORM\Column(name: 'vat_rate', type: 'smallint', nullable: true)]
	protected ?int $vatRate = null;

	#[ORM\Column(name: 'tracking_number', type: 'string', length: 255, nullable: true)]
	public ?string $trackingNumber = null;

	#[ORM\Column(name: 'tracking_url', type: 'string', length: 255, nullable: true)]
	public ?string $trackingUrl = null;

	public function __construct(?Spedition $spedition, Order $order)
	{
		$this->spedition = $spedition;
		$this->name      = $spedition ? $spedition->getName() : '';
		if ($spedition)
			$this->price = $spedition->getPrice();
		$this->order = $order;
	}

	public function getIdent(): ?string { return $this->spedition ? $this->spedition->getIdent() : null; }

	public function getSpeditionId(): ?int { return $this->spedition ? $this->spedition->getId() : null; }

	public function getSpedition(): ?Spedition { return $this->spedition; }

	public function setSpedition(Spedition $spedition): self
	{
		$this->spedition = $spedition;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		if ($this->getOrder() && $this->getOrder()->isZeroVat()) {
			$result = $this->getPriceWithoutVat($useCurrency);
		} else {
			$result = $useCurrency
				? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
				: (float) $this->price;
		}

		if ($useCurrency) {
			$result = round($result, $this->getOrder()->currency->decimals ?? 2);
		}

		return $result;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$price = $useCurrency
			? $this->getOrder()->calculateCurrencyPrice((float) $this->price)
			: (float) $this->price;

		$price = OrderHelper::removeVat($price, $this->vatRate);

		if ($useCurrency) {
			$price = round($price, $this->getOrder()->currency->decimals ?? 2);
		} else {
			$price = round($price, 2);
		}

		return $price;
	}

	public function setPrice(float $price): self
	{
		$this->price = Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getOrder(): Order { return $this->order; }

	public function getPickupTypeTitle(): ?string
	{
		if ($this->getSpedition()
			&& in_array($this->getSpedition()->getIdent(), ['zasilkovna', 'ulozenka', 'dpdPickup', 'balikomat']))
			return $this->getName();

		return null;
	}

	public function setVatRate(?int $vatRate): void { $this->vatRate = $vatRate; }

	public function getVatRate(): int { return $this->getOrder()->isZeroVat() ? 0 : ($this->vatRate ?: 21); }
}
