<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\SellerInvoiceConfig;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use function str_ends_with;

class InvoiceConfigRepository extends BaseEntityService
{
	/** @var string */
	protected $entityClass = InvoiceConfig::class;

	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('ic');
	}

	/**
	 * @throws Exception
	 */
	public function getDueDate(int $sellerId): ?DateTime
	{
		$now           = new DateTime;
		$defaultConfig = $this->getConfigBySeller($sellerId);

		if (!$defaultConfig) {
			return null;
		}

		return $now->modify(sprintf('+ %s days', $defaultConfig->maturity));
	}

	/**
	 * @throws NonUniqueResultException
	 */
	public function generateIdent(int $sellerId, string $type = InvoiceConfig::TYPE_INVOICE): ?string
	{
		$defaultConfig = $this->getConfigBySeller($sellerId);
		if (!$defaultConfig) {
			return null;
		}

		$numericalSeries = match ($type) {
			InvoiceConfig::TYPE_INVOICE => $defaultConfig->numericalSeries,
			InvoiceConfig::TYPE_CORRECTIVE => $defaultConfig->correctiveNumericalSeries,
			InvoiceConfig::TYPE_RECEIPT => $defaultConfig->receiptNumericalSeries,
			default => throw new InvalidArgumentException('Value in $type arg is unknown'),
		};

		$containsPrefixYearWildcards  = $numericalSeries->containsPrefixYearWildcards();
		$containsPrefixMonthWildcards = $numericalSeries->containsPrefixMonthWildcards();
		$startNumber                  = $numericalSeries->startNumber;
		$getLastDocumentData          = function() use ($type) {
			switch ($type) {
				case InvoiceConfig::TYPE_INVOICE:
				case InvoiceConfig::TYPE_CORRECTIVE:
					$qb = $this->em->getRepository(Invoice::class)->createQueryBuilder('i');
					$qb->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
						->andWhere('o.isCorrectiveTaxDocument = :isCorrectiveTaxDocument')
						->orderBy('i.createdAt', 'desc')
						->setParameter('isCorrectiveTaxDocument', (int) ($type === InvoiceConfig::TYPE_CORRECTIVE))
						->setMaxResults(1);

					/** @var Invoice|null $lastInvoice */
					$lastInvoice = $qb->getQuery()->getOneOrNullResult();
					if (!$lastInvoice) {
						return null;
					}

					return ['ident' => $lastInvoice->ident, 'createdAt' => $lastInvoice->createdAt];
				case InvoiceConfig::TYPE_RECEIPT:
					$qb = $this->em->getRepository(Order::class)->createQueryBuilder('o');
					$qb->where('o.receiptIdent IS NOT NULL')
						->orderBy('o.receiptIdent', 'desc')
						->setMaxResults(1);

					/** @var Order|null $lastReceiptOrder */
					$lastReceiptOrder = $qb->getQuery()->getOneOrNullResult();

					if (!$lastReceiptOrder) {
						return null;
					}

					return ['ident'     => $lastReceiptOrder->receiptIdent,
					        'createdAt' => $lastReceiptOrder->getCreatedTime()];
				default:
					throw new InvalidArgumentException('Value in $type arg is unknown');
			}
		};

		if ($containsPrefixMonthWildcards || $containsPrefixYearWildcards) {
			$lastDocumentData = $getLastDocumentData();

			if ($lastDocumentData !== null) {
				$created = $lastDocumentData['createdAt'];
				$now     = new DateTime;

				// prefix contain month/year wildcards and current month/year is bigger then on the invoice
				if (($containsPrefixMonthWildcards && (((int) $now->format('n')) > ((int) $created->format('n')))) ||
					($containsPrefixYearWildcards && (((int) $now->format('Y')) > ((int) $created->format('Y'))))) {
					$this->resetInvoiceCounter($sellerId);
					$startNumber = 1;
				}
			}
		}

		$strMaxOrder = Strings::padRight('9', $numericalSeries->digitsCount, '9');

		if (((int) $strMaxOrder) === $startNumber) {
			$lastDocumentData = $getLastDocumentData();

			if (str_ends_with($lastDocumentData['ident'], $strMaxOrder)) {
				return null;
			}
		}

		$prefix = $numericalSeries->getRealPrefix();
		$order  = Strings::padLeft((string) $startNumber, $numericalSeries->digitsCount, '0');

		return sprintf('%s%s', $prefix, $order);
	}

	/**
	 *
	 * @throws NonUniqueResultException
	 */
	private function resetInvoiceCounter(int $sellerId, string $type = InvoiceConfig::TYPE_INVOICE): void
	{
		$defaultConfig = $this->getConfigBySeller($sellerId);

		if (!$defaultConfig) {
			return;
		}

		$numericalSeries = match ($type) {
			InvoiceConfig::TYPE_INVOICE => $defaultConfig->numericalSeries,
			InvoiceConfig::TYPE_CORRECTIVE => $defaultConfig->correctiveNumericalSeries,
			InvoiceConfig::TYPE_RECEIPT => $defaultConfig->receiptNumericalSeries,
			default => throw new InvalidArgumentException('Value in $type arg is unknown'),
		};

		$numericalSeries->startNumber = 1;

		$this->em->persist($numericalSeries);
		$this->em->flush();
	}

	/**
	 * @return SellerInvoiceConfig[]
	 */
	public function getUsed(): array
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('sic')
			->from(SellerInvoiceConfig::class, 'sic')
			->join('sic.seller', 's');

		return $qb->getQuery()->getResult();
	}

	public function haveAllSellersInvoiceSettingsCreated(): bool
	{
		return $this->em->getRepository(Seller::class)->count([]) === count($this->getUsed());
	}

	/**
	 * @throws NonUniqueResultException
	 */
	public function getConfigBySeller(int $sellerId): ?InvoiceConfig
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('ic')
			->from(InvoiceConfig::class, 'ic')
			->join('ic.sellerInvoiceConfigs', 'sic')
			->join('sic.seller', 's')
			->where('s.id = :id')
			->setParameter('id', $sellerId)
			->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}

}
