<?php declare(strict_types = 1);

namespace EshopOrders\Model\Listeners;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Doctrine\ORM\Event\PostPersistEventArgs;
use Doctrine\ORM\Event\PostUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\CarrierHistory;
use EshopOrders\Model\Entities\Order;
use Exception;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class CarrierListener implements EventSubscriberInterface
{
	protected static bool $saved = false;

	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'orderSubscriber.orderSpeditionFormSuccess' => ['speditionChanged'],
		];
	}

	/**
	 * @param mixed $entity hlavni entita z dopravcu (UlozenkaOrder, atd.)
	 *
	 * @throws Exception
	 */
	#[ORM\PostPersist]
	#[ORM\PostUpdate]
	public function postHandler(mixed $entity, PostPersistEventArgs|PostUpdateEventArgs $event): void
	{
		$em = $event->getObjectManager();

		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($entity);

		if ($changeSet && isset($changeSet['numberPackage'])) {
			/** @var Order $order */
			$order = $entity->getOrder();

			$packages = [];
			if (method_exists($entity, 'getAllNumberPackages')) {
				$packages = $entity->getAllNumberPackages();
			} else {
				$packages[] = $entity->numberPackage;
			}

			$this->speditionChanged(new Event(['order' => $order, 'numberPackages' => $packages]));
			static::$saved = true;
		}
	}

	public function speditionChanged(Event $event): void
	{
		if (static::$saved) {
			return;
		}

		$this->save($event->data['numberPackages'], $event->data['order']);
		static::$saved = true;
	}

	protected function save(array $packages, Order $order): void
	{
		$ch = new CarrierHistory(
			$order, $order->getSpedition()
			->getName(), $packages ? implode(', ', $packages) : null
		);

		try {
			$this->em->persist($ch);
			$this->em->flush($ch);
		} catch (Exception $e) {
			Debugger::log($e, 'carrier-listener');
		}
	}

}
