<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderItemReview;
use EshopOrders\Model\Entities\OrderStatus;
use Exception;
use Nette\Utils\DateTime;
use Tracy\Debugger;

/**
 * @method OrderItemReview|null getReference($id)
 * @method OrderItemReview[]|null getAll()
 * @method OrderItemReview|null get($id)
 */
class OrderItemReviews extends BaseEntityService
{
	protected $entityClass = OrderItemReview::class;

	public function __construct(protected EventDispatcher $eventDispatcher)
	{
	}

	public function sendRequestsForReview(): void
	{
		if (!EshopOrdersConfig::load('orderItemReviews.enable')) {
			return;
		}

		$afterDays = EshopOrdersConfig::load('orderItemReviews.sendRequestForReview.afterDays');
		$inStatus  = EshopOrdersConfig::load(
			'orderItemReviews.sendRequestForReview.inStatus',
			OrderStatus::STATUS_FINISHED,
		);
		$from      = (new DateTime(sprintf('-%s days', $afterDays)))->setTime(0, 0, 0);
		$to        = (new DateTime(sprintf('-%s days', $afterDays)))->setTime(23, 59, 59);

		$qb = $this->em->getRepository(Order::class)->createQueryBuilder('o');
		$qb->join('o.orderStatuses', 'oss')
			->join('oss.status', 'os', 'WITH', 'os.id = :status')
			->andWhere('oss.created BETWEEN :from AND :to')
			->setParameters([
				'status' => $inStatus,
				'from'   => $from,
				'to'     => $to,
			]);
		$orders = $qb->getQuery()->getResult();

		$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/requestForReview.latte';
		$originFile = __DIR__ . '/../FrontModule/_emails/requestForReview.latte';

		try {
			/** @var Order $order */
			foreach ($orders as $order) {
				if ($order->hasFlag(OrderFlag::TYPE_REQUEST_FOR_REVIEW)) {
					continue;
				}

				$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendRequestsForReview');

				$flag = new OrderFlag(OrderFlag::TYPE_REQUEST_FOR_REVIEW, '1', $order);
				$order->addFlag($flag);

				$this->em->persist($flag);
			}

			$this->em->flush();
		} catch (Exception $e) {
			Debugger::log($e);
		}
	}
}
