<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Exception;
use Nette\Caching\Cache;
use Tracy\Debugger;

/**
 * @method Payment|object|null getReference($id)
 * @method Payment|null get($id)
 */
class Payments extends BaseEntityService
{
	protected $entityClass = Payment::class;

	protected ?array $cForSelectOption = null;
	protected ?array $cByIdent         = null;

	public function __construct(protected EshopOrdersCache $eshopOrdersCache)
	{
	}

	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish($id, $state)
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function getAll()
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->orderBy('p.position');
		$payments = $paymentsQuery->getQuery()->getResult();

		return $payments;
	}

	public function getAllPublished()
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsQuery->orderBy('p.position');
		$payments = $paymentsQuery->getQuery()->getResult();

		return $payments;
	}

	/**
	 * @param int $cartValue cena objednavky
	 *
	 * @return Dao\Payment[] vsechny dopravy, ktere jsou aktivni, a jsou dostupne pro danou cenu objednavky
	 */
	public function getAllByCartValue($cartValue)
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsQuery->andWhere('p.availableFrom IS NULL OR p.availableFrom <= :cartValue')
			->setParameter('cartValue', $cartValue);
		$paymentsQuery->andWhere('p.availableTo IS NULL OR p.availableTo >= :cartValue')
			->setParameter('cartValue', $cartValue);
		$paymentsQuery->orderBy('p.position');
		$payments = $paymentsQuery->getQuery()->getResult();

		return $payments;
	}

	/**
	 *
	 * @return array{
	 *     name: ?string,
	 *     ident: ?string,
	 *     text: ?string,
	 *     isPublished: int,
	 *     position: int,
	 *     price: float,
	 *     currency: ?string,
	 *     freeFrom: ?int,
	 *     availableFrom: ?int,
	 *     availableTo: ?int,
	 *     image: ?string,
	 *     disableInGoogle: ?int,
	 *     pohodaName: ?string,
	 *     id: int,
	 * }
	 */
	public function getByIdent(string $ident, bool $onlyActive = false): ?array
	{
		if ($this->cByIdent === null) {
			$this->cByIdent = [];

			foreach ($this->getEr()->createQueryBuilder('p')
				         ->getQuery()->getArrayResult() as $row) {
				$row['price'] = (float) $row['price'];

				$this->cByIdent[$row['ident']] = $row;
			}
		}

		$result = $this->cByIdent[$ident] ?? null;

		if ($result && $onlyActive && !$result['isPublished']) {
			return null;
		}

		return $result;
	}

	public function getReferenceByIdent($ident)
	{
		return $this->getReference($this->getByIdent($ident)['id']);
	}

	public function getForSelectOption(): array
	{
		if ($this->cForSelectOption === null) {
			$this->cForSelectOption = [null => ''] + ($this->getEr()->findPairs([], 'name') ?: []);
		}

		return $this->cForSelectOption;
	}

	public function remove($id)
	{
		$result = parent::remove($id);

		if ($result) {
			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['payment', 'freeFrom'],
			]);
		}

		return $result;
	}

	public function setOrderPayment(
		Order   $order,
		Payment $payment,
		?string $name = null,
		?float  $price = null,
	): void
	{
		try {
			$orderPayment = $order->getPayment();

			if (!$orderPayment) {
				if (!$name) {
					$name = $payment->getName();
				}

				if (!$price) {
					$price = $payment->getPrice();
				}
			}

			$orderPayment->setPayment($payment);

			if ($price !== null) {
				$orderPayment->setPrice($price);
			}

			if ($name) {
				$orderPayment->setName($name);
			}

			$this->em->persist($orderPayment);
			$this->em->flush();
		} catch (Exception $e) {
			Debugger::log($e, 'eshoporders');

			throw $e;
		}
	}
}
