<?php declare(strict_types = 1);

namespace EshopOrders\Model\Statistics;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\QueryBuilder;
use DateTimeInterface;
use EshopCatalog\Model\Config;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopStock\Model\Entities\SupplyProduct;
use Nette\Utils\DateTime;

class BaseStatistics
{
	public function __construct(protected EntityManagerDecorator $em)
	{
	}

	public function getBaseOrdersQB(
		DateTimeInterface $from,
		DateTimeInterface $to,
		?string            $siteIdent = null,
	): QueryBuilder
	{
		$qb = $this->em->createQueryBuilder();

		$osCratedQb = $this->em->createQueryBuilder()->select('IDENTITY(os.order)')
			->from(OrderStatus::class, 'os')
			->where('os.status = :statusCrea')
			->andWhere('os.created >= :from')
			->andWhere('os.created <= :to');

		$query = $this->em->createQueryBuilder()
			->select(
				'o.id, p.price as paymentPrice, s.price as speditionPrice, IDENTITY(o.site) as site, IDENTITY(o.customer) as customer, curr.code as currencyCode, curr.rate as currencyRate, curr.decimals as currencyDecimals, IDENTITY(o.site) as siteIdent',
			)
			->from(Order::class, 'o')
			->join('o.payment', 'p')
			->join('o.spedition', 's')
			->leftJoin('o.currency', 'curr')
			->where($qb->expr()->in('o.id', $osCratedQb->getDQL()))
			->setParameters([
				'statusCrea' => OrderStatus::STATUS_CREATED,
				'from'       => $from,
				'to'         => $to,
			]);

		if ($siteIdent)
			$query->andWhere('o.site = :site')
				->setParameter('site', $siteIdent);

		return $query;
	}

	public function getSales(array $ids, bool $setAbsolute = false): array
	{
		$sales = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(od.order) as order, od.price')
				         ->from(OrderDiscount::class, 'od')
				         ->where('od.order IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				$price                  = $setAbsolute ? abs((float) $row['price']) : (float) $row['price'];
				$sales[$row['order']][] = $price;
			}
		}

		return $sales;
	}

	public function getOrderItems(array $ids, $qbCallback = null, $dataCallback = null): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			$stockData = [];
			if (class_exists('EshopStock\DI\EshopStockExtension') && Config::load('enablePurchasePrice')) {
				foreach ($this->em->createQueryBuilder()
					         ->select('IDENTITY(sp.product) as product, IDENTITY(sp.order) as order, sp.purchasePrice')
					         ->from(SupplyProduct::class, 'sp')
					         ->where('sp.order IN (' . implode(',', $chunk) . ')')
					         ->getQuery()
					         ->getScalarResult() as $row) {
					$stockData[$row['order']][$row['product']] = (float) $row['purchasePrice'];
				}
			}

			$qb = $this->em->createQueryBuilder()->select(
				'IDENTITY(oi.order) as order, oi.quantity, oi.price, oi.vatRate, IDENTITY(oi.product) as product,
			    s.value as saleValue, s.type as saleType',
			)
				->from(OrderItem::class, 'oi')
				->where('oi.order IN (' . implode(',', $chunk) . ')')
				->leftJoin('oi.sales', 's');

			if (Config::load('enablePurchasePrice')) {
				$qb->addSelect('p.purchasePrice')
					->leftJoin('oi.product', 'p');
			}

			if ($qbCallback) {
				$qbCallback($qb);
			}

			foreach ($qb->getQuery()->getScalarResult() as $row) {
				$purchasePrice = $stockData[$row['order']][$row['product']] ?? null;
				if ($purchasePrice === null && isset($row['purchasePrice'])) {
					$purchasePrice = (float) $row['purchasePrice'];
				}

				$item = [
					'price'         => (float) $row['price'],
					'purchasePrice' => $purchasePrice ?: 0,
					'quantity'      => (int) $row['quantity'],
					'vatRate'       => (int) $row['vatRate'],
					'saleValue'     => $row['saleValue'],
					'saleType'      => $row['saleType'],
				];

				if ($dataCallback) {
					$dataCallback($item, $row, $items);
				}

				$items[$row['order']][] = $item;
			}
		}

		return $items;
	}

	public function getPayments(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(op.order) as order, op.price, op.vatRate')
				         ->from(OrderPayment::class, 'op')
				         ->where('op.order IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				$items[$row['order']][] = [
					'price'   => (float) $row['price'],
					'vatRate' => (int) $row['vatRate'],
				];
			}
		}

		return $items;
	}

	public function getSpeditions(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(os.order) as order, os.price, os.vatRate')
				         ->from(OrderSpedition::class, 'os')
				         ->where('os.order IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getScalarResult() as $row) {
				$items[$row['order']][] = [
					'price'   => (float) $row['price'],
					'vatRate' => (int) $row['vatRate'],
				];
			}
		}

		return $items;
	}

	public function getCancelled(array $ids): array
	{
		$items = [];

		foreach (array_chunk($ids, 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('IDENTITY(os.order) as order')
				         ->from(OrderStatus::class, 'os')
				         ->where('os.order IN (' . implode(',', $chunk) . ')')
				         ->andWhere('os.status = :status')
				         ->setParameters([
					         'status' => OrderStatus::STATUS_CANCELED,
				         ])->getQuery()->getScalarResult() as $row) {
				$items[$row['order']] = $row['order'];
			}
		}

		return $items;
	}

	/**
	 *
	 * @return DateTimeInterface[]
	 */
	public function getCreated(DateTimeInterface $from, DateTimeInterface $to): array
	{
		$dates = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(os.order) as order, os.created')
			         ->from(OrderStatus::class, 'os')
			         ->andWhere('os.status = :statusCrea')
			         ->andWhere('os.created >= :from')
			         ->andWhere('os.created <= :to')
			         ->setParameters([
				         'statusCrea' => OrderStatus::STATUS_CREATED,
				         'from'       => $from,
				         'to'         => $to,
			         ])->getQuery()->getScalarResult() as $row) {
			$dates[$row['order']] = DateTime::createFromFormat('Y-m-d H:i:s', $row['created']);
		}

		return $dates;
	}
}
