<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Payment;

use Core\Model\Parameters;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Payments;

class PaymentsGrid extends BaseControl
{
	protected Payments            $paymentsService;
	protected IPaymentFormFactory $formFactory;

	public function __construct(
		Payments            $payments,
		IPaymentFormFactory $formFactory
	)
	{
		$this->paymentsService = $payments;
		$this->formFactory     = $formFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		if ($this->paymentsService->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.removeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.addPayment');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this->template->modalTitle = $this->t('eshopOrders.title.editPayment');
		$this['form']->setPayment($id);
		$this->template->modal = 'form';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->paymentsService->getEr()->createQueryBuilder('p')->orderBy('p.position', 'ASC');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('paymentsGrid:gridSortableRow!');

		//Columns
		$grid->addColumnText('name', 'default.name');
		$grid->addColumnText('ident', 'eshopOrders.paymentForm.itemIdent');
		$grid->addColumnStatus('isPublished', 'default.isActive')->setAlign('center')
			->addOption(1, 'eshopOrders.defaultGrid.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'eshopOrders.defaultGrid.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		$columnPrice = $grid->addColumnNumber('price', 'eshopOrders.default.priceSimple');
		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$grid->addColumnNumber('freeFrom', 'eshopOrders.default.freeFromSimple');
		}

		$grid->addColumnNumber('availableFrom', 'eshopOrders.default.availableFromSimple');
		$grid->addColumnNumber('availableTo', 'eshopOrders.default.availableToSimple');

		if (Parameters::load('currency.default', 'CZK') !== 'CZK') {
			$columnPrice->setFormat(2, ',', ' ');
		}

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		return $grid;
	}

	protected function createComponentForm(): PaymentForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setPayment((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[] = function() {
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};

		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id') ?: null;
		$position  = $request->getPost('position') ?: null;

		if ($id !== null && $position !== null && $this->paymentsService->setPosition((int) $id, (int) $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
		$presenter->redrawControl('paymentsGrid');
		$presenter->redrawControl('paymentSpeditionsGrid');
	}

	public function gridPublishChange(string $id, string $newStatus): void
	{
		if ($this->paymentsService->setPublish((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->redrawItem($id);
		$this->presenter->redrawControl('flashes');
	}
}
