<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use Doctrine\Common\Collections\ArrayCollection;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\OrderFacade;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;

class OrdersList extends BaseControl
{
	protected Orders      $ordersService;
	protected Customers   $customersService;
	protected OrderFacade $orderFacade;

	public function __construct(
		Orders      $orders,
		Customers   $customers,
		OrderFacade $orderFacade
	)
	{
		$this->ordersService    = $orders;
		$this->customersService = $customers;
		$this->orderFacade      = $orderFacade;
	}

	public function render(): void
	{
		$this->template->setFile($this->getTemplateFile());

		$userId   = (int) $this->getPresenter()->getUser()->id;
		$customer = $this->customersService->getByUserId($userId);
		if ($customer) {
			$orders                 = $this->ordersService->getByCustomer($customer);
			$this->template->orders = $orders;
		} else {
			$this->template->orders = new ArrayCollection();
		}

		$this->template->showUser = EshopOrdersConfig::load('showOrdersOfChildrenUsers');

		$this->template->render($this->getTemplateFile());
	}

	public function handleRepeatOrder(int $id): void
	{
		$presenter = $this->presenter;

		$result = $this->orderFacade->repeatOrder($id);
		if ($result['added'] === 0) {
			$presenter->flashMessageDanger('eshopOrdersFront.list.repeatOrderNoProducts');
		} else if ($result['notQuantity'] > 0) {
			$presenter->flashMessageInfo('eshopOrdersFront.list.repeatOrderNoQuantity');
		} else {
			$presenter->flashMessageSuccess('eshopOrdersFront.list.repeatOrderSuccess');
		}

		if ($result['status'] === 'ok') {
			$event = new Event(['orderId' => $id]);
			$this->eventDispatcher->dispatch($event, 'eshopOrders.repeatOrder');

			$presenter->redirect(':EshopOrders:Front:Default:order');
		} else {
			if ($presenter->isAjax()) {
				$presenter->redrawControl('flashes');
			} else {
				$presenter->redirect('this');
			}
		}
	}
}

