<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Countries;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Forms\Form;
use Nette\Forms\ISubmitterControl;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;

class ProfileForm extends BaseControl
{
	public User $user;

	protected Customer  $customer;
	protected Customers $customers;
	protected Countries $countriesService;

	public function __construct(
		User      $user,
		Customers $customers,
		Countries $countries
	)
	{
		$this->customers        = $customers;
		$this->countriesService = $countries;
		$this->user             = $user;
		$this->customer         = $this->customers->getOrCreateCustomer($user->getId(), $user->getEmail(), $user->getName(), $user->getLastname());;
	}

	public function render(): void
	{
		if (EshopOrdersConfig::load('allowAddressManagement', false)) {
			$this->template->addressesOptions = $this->getAddressesOptions();
		}

		$this->template->customer = $this->customer;
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('name', 'default.name');
		$form->addText('lastname', 'default.lastName')
			->setRequired();
		$form->addPassword('password', 'default.password')
			->setRequired(false);
		$form->addEmail('email', 'default.email')
			->setRequired();
		$form->addText('phone', 'default.phone');

		$customer = $this->customer;
		$user     = $this->user;
		$d        = [
			'name'     => $user->getName(),
			'lastname' => $user->getLastname(),
			'email'    => $user->getEmail(),
			'phone'    => $customer->getPhone(),
		];

		if (!EshopOrdersConfig::load('allowAddressManagement', false)) {
			$countries = $this->countriesService->getAllNameColumn();

			$addressDelivery = $form->addContainer('addressDelivery');
			$addressInvoice  = $form->addContainer('addressInvoice');

			if ($customer->getAddressDelivery() && $customer->getAddressInvoice()
				&& $customer->getAddressDelivery()->getId() === $customer->getAddressInvoice()->getId()) {
				$tmp = new CustomerAddress($customer);
				$customer->setAddressDelivery($tmp);
				$this->em->persist($tmp);
				$this->em->persist($customer);
				$this->em->flush();
			}

			$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal', 'idNumber',
				'vatNumber'];
			if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
				$fields[] = 'idVatNumber';
			}
			$fields[]         = 'country';
			$disableModifyIco = EshopOrdersConfig::load('customer.disableModifyIco');

			foreach ($fields as $v) {
				foreach (['addressDelivery', 'addressInvoice'] as $com) {
					if ($v === 'email') {
						${$com}->addEmail($v, 'eshopOrders.default.' . $v)
							->setMaxLength(255);
					} else if ($v === 'country') {
						${$com}->addSelect($v, 'eshopOrders.default.' . $v, $countries);
					} else {
						${$com}->addText($v, 'eshopOrders.default.' . $v)
							->setMaxLength(255);
					}

					if ($disableModifyIco && Arrays::contains(['idNumber', 'vatNumber'], $v)) {
						${$com}[$v]->setDisabled();
					}

					if ($customer->$com) {
						if ($v === 'country') {
							/** @phpstan-ignore-next-line */
							$d[$com][$v] = $customer->$com->$v ? $customer->$com->$v->getId() : null;
						} else {
							/** @phpstan-ignore-next-line */
							$d[$com][$v] = $customer->$com->$v;
						}
					}
				}
			}
		} else {
			$addresses = [];
			foreach ($customer->addressesAvailable as $addr) {
				if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')
					&& $customer->getAddressInvoice()
					&& $customer->getAddressInvoice()->getId() === $addr->getId()
				) {
					continue;
				}

				$addresses[$addr->getId()] = $addr->getDetailString();
			}

			$form->addSelect('addressDelivery', 'eshopOrdersFront.customer.defaultAddressDelivery', $addresses);

			if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				$form->addSelect('addressInvoice', 'eshopOrdersFront.customer.defaultAddressInvoice', $addresses);
			}
		}

		$form->addSubmit('submit', 'default.save')
			->setBtnClass('btn-success primary-bg-color');
		$form->addCancel('cancel', 'default.cancel')
			->setBtnClass('btn-secondary secondary-bg-color');

		$form->onSuccess[] = [$this, 'formSuccess'];
		$form->setDefaults($d);

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values): bool
	{
		if (
			$form->isSubmitted() instanceof ISubmitterControl
			&& method_exists($form->isSubmitted(), 'getName')
			&& $form->isSubmitted()->getName() === 'cancel'
		) {
			$this->presenter->redirect(':EshopOrders:Front:Customer:');
		}

		try {
			$user     = $this->user;
			$customer = $this->customer;

			$user->setName($values->name);
			$user->setLastname($values->lastname);
			$user->setEmail($values->email);
			$customer->setPhone($values->phone);

			if (!EshopOrdersConfig::load('allowAddressManagement', false)) {
				$addressDelivery = $customer->getAddressDelivery() ?: new CustomerAddress($customer);
				$addressInvoince = $customer->getAddressInvoice() ?: new CustomerAddress($customer);

				$customer->setAddressDelivery($addressDelivery);
				$customer->setAddressInvoice($addressInvoince);

				$fields = ['firstName', 'lastName', 'email', 'phone', 'company', 'street', 'city', 'postal'];

				if (!EshopOrdersConfig::load('customer.disableModifyIco')) {
					$fields[] = 'idNumber';
					$fields[] = 'vatNumber';
				}

				if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
					$fields[] = 'idVatNumber';
				}
				$fields[] = 'country';

				foreach ($fields as $v) {
					foreach (['addressDelivery', 'addressInvoice'] as $com) {
						try {
							if ($v == 'country') {
								if ($values->$com->$v)
									$customer->$com->$v = $this->countriesService->getReference($values->$com->$v);
							} else {
								$customer->$com->$v = $values->$com->$v;
							}
						} catch (\Exception $e) {
						}
					}
				}

				$this->em->persist($addressDelivery);
				$this->em->persist($addressInvoince);
			} else {
				/** @var CustomerAddress|null $addrRef */
				$addrRef = $values->addressDelivery ? $this->em->getReference(CustomerAddress::class, $values->addressDelivery) : null;

				$customer->setAddressDelivery($addrRef);

				if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					/** @var CustomerAddress|null $addrRef */
					$addrRef = $values->addressInvoice ? $this->em->getReference(CustomerAddress::class, $values->addressInvoice) : null;

					$customer->setAddressInvoice($addrRef);
				}
			}

			if ($values->password != '') {
				$user->setPassword($values->password);
			}

			$this->em->persist($user);
			$this->em->persist($customer);
			$this->em->flush();
			$this->presenter->flashMessageSuccess('users.profileForm.profileUpdated');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function getAddressesOptions(): array
	{
		$addresses = [
			'delivery' => '',
			'invoice'  => '',
		];

		$customer = $this->customer;

		$deliveryDefault = $customer->getAddressDelivery() ? $this->customer->getAddressDelivery()->getId() : null;
		$invoiceDefault  = $customer->getAddressInvoice() ? $this->customer->getAddressInvoice()->getId() : null;

		foreach ($this->customer->addressesAvailable as $addr) {
			if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')
				&& $invoiceDefault === $addr->getId()) {
				continue;
			}

			$str                   = $addr->getDetailString();
			$addresses['delivery'] .= '<option value="' . $addr->getId() . '"' . ($deliveryDefault === $addr->getId() ? ' selected' : '') . '>' . $addr->getDetailString() . '</option>';
			$addresses['invoice']  .= '<option value="' . $addr->getId() . '"' . ($invoiceDefault === $addr->getId() ? ' selected' : '') . '>' . $addr->getDetailString() . '</option>';
		}

		return $addresses;
	}
}

