<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\FrontModule\Model\Event\PaymentsSpeditionsEvent;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\Entities\SpeditionPriceLevel;
use EshopOrders\Model\Entities\SpeditionWeight;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Users\Model\Security\User;

class Speditions extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Spedition::class;

	public EventDispatcher     $eventDispatcher;
	public Sites               $sitesService;
	protected User             $user;
	protected Customers        $customers;
	protected EshopOrdersCache $eshopOrdersCache;

	/** @var Dao\Spedition[] */
	protected ?array $cPublished = null;

	/** @var Dao\SpeditionWeight[][] */
	protected ?array $cWeights = null;

	/** @var SpeditionPriceLevel[]|null */
	protected ?array $cGroupPriceLevels = null;

	public function __construct(
		EventDispatcher  $eventDispatcher,
		Sites            $sitesService,
		User             $user,
		Customers        $customers,
		EshopOrdersCache $eshopOrdersCache
	)
	{
		$this->eventDispatcher  = $eventDispatcher;
		$this->sitesService     = $sitesService;
		$this->user             = $user;
		$this->customers        = $customers;
		$this->eshopOrdersCache = $eshopOrdersCache;
	}

	public function get(int $id): ?Dao\Spedition
	{
		return $this->getAllPublished()[$id];
	}

	/**
	 * @return Dao\SpeditionWeight[][]
	 */
	public function getWeights(): array
	{
		if ($this->cWeights === null) {
			$this->cWeights = $this->eshopOrdersCache->getCache()->load('speditionWeights', function(&$dep): array {
				$dep = [
					Cache::EXPIRATION => '1 week',
				];

				$weights = [];
				foreach ($this->em->getRepository(SpeditionWeight::class)->createQueryBuilder('sw')
					         ->orderBy('sw.weightFrom', 'ASC')
					         ->getQuery()->getArrayResult() as $row) {
					$weightFrom = (float) $row['weightFrom'];

					$price = (float) $row['price'];
					if (EshopOrdersConfig::load('speditions.priceIsWithoutVat')) {
						$price = round($price * 1.21, 2);
					}

					$weights[$row['id']][(string) $weightFrom] = new Dao\SpeditionWeight($weightFrom, $price);
				}

				return $weights;
			});
		}

		return $this->cWeights;
	}

	/**
	 * @return Dao\Spedition[]
	 */
	public function getAllPublished(): array
	{
		if ($this->cPublished === null) {
			$this->cPublished = [];
			$qb               = $this->getEr()->createQueryBuilder('s', 's.id')
				->andWhere('s.isPublished = :isPublished')
				->innerJoin(PaymentSpedition::class, 'ps', Join::WITH, 'ps.spedition = s.id')
				->leftJoin('s.vatRate', 'vr')
				->addSelect('vr.rate as vatRate')
				->setParameters([
					'isPublished' => 1,
				])
				->orderBy('s.position')
				->groupBy('s.id');

			if (count($this->sitesService->getSites()) > 1) {
				$qb->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
					->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());
			}

			$customer = $this->user->isLoggedIn() && $this->user->getId() ? $this->customers->getByUserId($this->user->getId()) : null;
			if ($customer) {
				if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer', false) && $customer->disabledSpeditions->getKeys()) {
					$qb->andWhere('s.id NOT IN (:notIds)')
						->setParameter('notIds', $customer->disabledSpeditions->getKeys());
				}

				if (EshopOrdersConfig::load('enablePriceLevels', false)) {
					$group = $customer->getGroupCustomers() ?: null;

					if ($group) {
						$qb->leftJoin('s.priceLevels', 'sPriceLevels', Join::WITH, 'sPriceLevels.group = :group')
							->setParameter('group', $group->getId())
							->addSelect('sPriceLevels');
					}
				}
			}

			if (EshopOrdersConfig::load('speditions.allowWeights')) {
				$qb->leftJoin('s.weights', 'weights')
					->addSelect('weights');
			}

			$result = $qb->getQuery()->getArrayResult();

			$this->cPublished = $this->fillDao($result);

			foreach ($this->cPublished as $id => $spedition) {
				if (EshopOrdersConfig::load('speditions.allowWeights')) {
					$this->cPublished[$id]->weights = $this->getWeights()[$id] ?? [];
				}
			}

			$this->eventDispatcher->dispatch(new PaymentsSpeditionsEvent(null, $this->cPublished), Speditions::class . '::afterFillDao');
		}

		return $this->cPublished;
	}

	/**
	 * @param Dao\Spedition[] $speditions
	 *
	 * @return Dao\Spedition[]
	 */
	public function filterByCart(array $speditions, Dao\Cart $cart): array
	{
		$cartValue        = $cart->getCartItemsPrice();
		$disabledPickUp   = $cart->hasDisabledPickUpSpedition();
		$oversizedProduct = $cart->hasOversizedProduct();
		$cartTotalWeight  = $cart->getTotalWeightInKG();

		foreach ($speditions as $k => $spedition) {
			if ($spedition->getAvailableFrom() > $cartValue || $spedition->getAvailableTo() < $cartValue) {
				unset($speditions[$k]);
			}

			if (
				($disabledPickUp && $spedition->isPickup === true)
				|| ($oversizedProduct && $spedition->allowOversized === false)
			) {
				unset($speditions[$k]);
			}

			if (EshopOrdersConfig::load('speditions.allowWeights')) {
				if ($spedition->maxWeight && $spedition->maxWeight < ($cartTotalWeight / 1000)) {
					unset($speditions[$k]);
				}
			}
		}

		return $speditions;
	}

	/**
	 * @return \Doctrine\ORM\Proxy\Proxy|Spedition|null
	 */
	public function getReference(int $id) { return $this->em->getReference($this->entityClass, $id); }

	/**
	 * @param array[] $speditionsRaw
	 *
	 * @return Dao\Spedition[]
	 */
	public function fillDao(array $speditionsRaw): array
	{
		$speditions = [];
		$now        = new DateTime();
		foreach ($speditionsRaw as $spedRaw) {
			$spedition = new Dao\Spedition();

			$spedition->vat = $spedRaw['vatRate'] ?: (int) EshopOrdersConfig::load('speditions.vat', 21);

			$spedRaw    = $spedRaw[0];
			$priceLevel = EshopOrdersConfig::load('enablePriceLevels', false) && $spedRaw['priceLevels']
				? array_values($spedRaw['priceLevels'])[0] ?? null
				: null;

			$spedition
				->setId((int) $spedRaw['id'])
				->setIdent($spedRaw['ident'])
				->setName($spedRaw['name'])
				->setText($spedRaw['text'])
				->setPublished((int) $spedRaw['isPublished'])
				->setPosition((int) $spedRaw['position'])
				->setAvailableFrom($priceLevel ? $priceLevel['availableFrom'] : $spedRaw['availableFrom'])
				->setAvailableTo($priceLevel ? $priceLevel['availableTo'] : $spedRaw['availableTo'])
				->setImage($spedRaw['image']);

			$price = (float) ($priceLevel ? $priceLevel['price'] : $spedRaw['price']);
			if (EshopOrdersConfig::load('speditions.priceIsWithoutVat')) {
				$price = round($price * (($spedition->vat / 100) + 1), 2);
			}
			$spedition->setPrice($price);

			$spedition->isStorePickup  = $spedRaw['isStorePickup'] ? true : false;
			$spedition->isPickup       = $spedRaw['isPickup'] ? true : false;
			$spedition->allowOversized = $spedRaw['allowOversized'] ? true : false;
			$spedition->code1          = $spedRaw['code1'];
			$spedition->code2          = $spedRaw['code2'];
			$spedition->zboziId        = $spedRaw['zboziId'];
			$spedition->heurekaId      = $spedRaw['heurekaId'];
			$spedition->googleId       = $spedRaw['googleId'];
			$spedition->idealoId       = $spedRaw['idealoId'];

			$spedition->priceInBaseCurrency = $spedition->getPrice();

			if (EshopOrdersConfig::load('allowFreeFrom')) {
				if ($priceLevel && $priceLevel['freeFrom'] !== null) {
					$spedition->setFreeFrom((float) $priceLevel['freeFrom']);
				} else if ($spedRaw['freeFrom'] !== null) {
					$spedition->setFreeFrom((float) $spedRaw['freeFrom']);
				}

				if ($priceLevel) {
					$spedition->freeFromDate = $priceLevel['freeFromDate'];
					$spedition->freeToDate   = $priceLevel['freeToDate'];
				} else {
					$spedition->freeFromDate = $spedRaw['freeFromDate'];
					$spedition->freeToDate   = $spedRaw['freeToDate'];
				}

				if (
					$spedition->freeFromDate && $spedition->freeToDate
					&& $spedition->freeFromDate <= $now && $spedition->freeToDate >= $now
				) {
					$spedition->setFreeFrom(0);
				}

				$spedition->freeFromInBaseCurrency = $spedition->getFreeFrom();
			}

			if (EshopOrdersConfig::load('speditions.allowWeights')) {
				$spedition->maxWeight = $spedRaw['maxWeight'] ? (float) $spedRaw['maxWeight'] : null;
			}

			$speditions[$spedition->getId()] = $spedition;
		}

		return $speditions;
	}

	/**
	 * @return SpeditionPriceLevel[]
	 */
	protected function getGroupPriceLevels(int $customerGroupId): array
	{
		if ($this->cGroupPriceLevels === null) {
			$this->cGroupPriceLevels = [];
			foreach ($this->getEr()->findBy(['groupId' => $customerGroupId]) as $row) {
				/** @var SpeditionPriceLevel $row */
				$this->cGroupPriceLevels[$row->getSpedition()->getId()] = $row;
			}
		}

		return $this->cGroupPriceLevels;
	}
}

