<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Table("eshop_orders__cart_item")
 * @ORM\Entity
 */
class CartItem
{
	use TId;

	/**
	 * Identifikator slozeny z id produktu, varianty, a pripadne dalsich parametru, ktere maji vliv na cenu
	 * @ORM\Column(name="ident", type="string", nullable=false)
	 */
	protected string $ident;

	/**
	 * @ORM\Column(name="name", type="string", length=255, nullable=true)
	 */
	public ?string $name = null;

	/**
	 * @var double
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=3, nullable=true)
	 */
	public $price = null;

	/**
	 * @ORM\Column(name="image", type="string", length=255, nullable=true)
	 */
	public ?string $image = null;

	/**
	 * @ORM\Column(name="product_id", type="integer", nullable=true)
	 */
	public ?int $productId = null;

	/**
	 * @ORM\Column(name="quantity", type="smallint", nullable=false, options={"default":1})
	 */
	public int $quantity;

	/**
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	public ?int $vatRate = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Cart", inversedBy="cartItems")
	 * @ORM\JoinColumn(name="cart_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Cart $cart;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	/**
	 * @var ArrayCollection<self>
	 * @ORM\OneToMany(targetEntity="CartItem", mappedBy="parent")
	 */
	protected Collection $children;

	/**
	 * @ORM\ManyToOne(targetEntity="CartItem", inversedBy="children")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?CartItem $parent = null;

	public function __construct(string $ident, ?int $productId, Cart $cart, int $quantity = 1)
	{
		$this->ident     = $ident;
		$this->productId = $productId;
		$this->quantity  = $quantity;
		$this->cart      = $cart;
		$this->moreData  = [];
		$this->children  = new ArrayCollection();

		//Při vytvoření nové položky košíku je potřeba ji přidat rovnou do Entity košíku. Jelikož Proxy Entity ji po vytvoření neobsahuje
		if ($ident) {
			$this->cart->cartItems->set($ident, $this);
		} else {
			$this->cart->addCartItem($this);
		}
	}

	public function getIdent(): string { return $this->ident; }

	public function getProductId(): ?int { return $this->productId; }

	public function getMoreData(): array { return $this->moreData ?? []; }

	/** @return mixed */
	public function getMoreDataValue(string $key)
	{
		return $this->moreData[$key] ?? null;
	}

	public function setMoreData(array $data): self
	{
		if ($this->moreData['note'] ?? null) {
			$data['note'] = $this->moreData['note'] . ($data['note'] ? "\n" . $data['note'] : '');
		}

		$this->moreData = $data;

		return $this;
	}

	/** @param mixed $value */
	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData)) {
			$this->moreData = [];
		}

		$this->moreData[$key] = $value;

		return $this;
	}

	/** @return Collection<self> */
	public function getChildren() { return $this->children; }

	public function getParent(): ?CartItem { return $this->parent; }

	public function setParent(CartItem $parent): void
	{
		$this->parent = $parent;
		$parent->getChildren()->set($this->id, $this);
	}
}
