<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities\Invoice;

use Core\Model\Entities\TId;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Table("eshop_orders__invoice_product")
 * @ORM\Entity
 */
class Product
{
	use TId;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $name = null;

	/**
	 * @ORM\Column(name="code1", type="string", nullable=true)
	 */
	public ?string $code1 = null;

	/**
	 * Parametr podstatny jen pri vytvareni faktury, jinak nepouzivat
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public ?int $productId;

	/**
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public ?int $parentId = null;

	/**
	 * @ORM\Column(type="integer", options={"default": 0})
	 */
	public int $quantity = 0;

	/**
	 * @var float|string
	 * @ORM\Column(type="decimal", precision=10, scale=2)
	 */
	public $price = 0;

	/**
	 * @ORM\Column(type="smallint")
	 */
	protected int $vatRate;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $imageBase64 = null;

	/**
	 * @ORM\ManyToOne(targetEntity="InvoiceData")
	 * @ORM\JoinColumn(name="invoice_data_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected InvoiceData $invoiceData;

	/**
	 * @var float|string|null
	 * @ORM\Column(name="recycling_fee", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $recyclingFee;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	public function __construct(InvoiceData $invoiceData)
	{
		$this->invoiceData = $invoiceData;
	}

	public function getImageBase64(): ?string
	{
		return $this->imageBase64 ? str_replace('png', 'jpeg', $this->imageBase64) : null;
	}

	public function getPrice(): float
	{
		return (float) $this->price;
	}

	public function getQuantity(): int { return $this->quantity ?: 0; }

	public function getPriceTotal(): float
	{
		return (float) $this->getPrice() * $this->quantity;
	}

	public function getTotalPriceWithoutVat(): float
	{
		return round($this->getPriceWithoutVat() * $this->quantity, 2);
	}

	public function getPriceWithoutVat(): float
	{
		$price = (float) $this->price;
		$vat = round($price - $price / (1 + ($this->vatRate / 100)), 2);

		return round($price - $vat, 2);
	}

	public function getVatRate(): int { return (int) $this->vatRate; }

	public function setVatRate(int $rate): void { $this->vatRate = $rate; }

	public function getRecyclingFeeWithoutVat(): float
	{
		return (float) $this->recyclingFee;
	}

	public function getTotalRecyclingFeeWithoutVat(): float
	{
		return (float) $this->recyclingFee * $this->quantity;
	}

	public function getMoreData(): array { return $this->moreData ?? []; }

	/** @return mixed */
	public function getMoreDataValue(string $key)
	{
		return $this->moreData[$key] ?? null;
	}

	public function setMoreData(array $data): self
	{
		$this->moreData = $data;

		return $this;
	}
}
