<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Currency\Model\Config;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use Nette\Utils\DateTime;

/**
 * @ORM\Table("eshop_orders__order_discount")
 * @ORM\Entity
 */
class OrderDiscount implements IDiscount
{
	public const TYPE_PERCENT        = 'percent';
	public const TYPE_FIX            = 'fix';
	public const TYPE_DELIVERY_PRICE = 'deliveryPrice';

	use TId;

	/**
	 * @ORM\Column(name="code", type="string")
	 */
	protected ?string $code = null;

	/**
	 * @ORM\Column(name="name", type="string")
	 */
	protected ?string $name = null;

	/**
	 * sleva v procentech nebo pevna castka
	 * @var int|float
	 * @ORM\Column(name="value", type="decimal", precision=10, scale=2)
	 */
	protected $value;

	/**
	 * vysledna sleva vypocitana z procent - zaporna hodnota
	 * @var float|string
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * procenta / fixni castka
	 *
	 * @ORM\Column(name="type", type="string")
	 */
	protected ?string $type = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderDiscounts")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Order $order;

	public function __construct(string $code, Order $order)
	{
		$this->code  = $code;
		$this->order = $order;
	}

	public function calculateDiscount(float $price, bool $useCurrency = false): float
	{
		return -abs($this->getPrice($useCurrency));
	}

	/*****
	 * === Code
	 */

	public function getCodeFormatted(): string
	{
		if (in_array((string) $this->code, ['ESHOPSALESAUTO', 'CHECKOUTSALE'])) {
			return '';
		}

		return (string) $this->code;
	}

	public function getCode(): ?string { return $this->code; }

	public function setCode(?string $code): OrderDiscount
	{
		$this->code = $code;

		return $this;
	}

	/*****
	 * === Name
	 */

	public function getName(): string { return $this->name; }

	public function setName(?string $name): OrderDiscount
	{
		$this->name = $name;

		return $this;
	}

	/*****
	 * === Quantity
	 */

	public function getQuantity(): int { return 1; }

	/*******
	 * === Value
	 */

	public function getValue(): float
	{
		return (float) $this->value;
	}

	public function setValue(float $value): self
	{
		$this->value = $value;

		return $this;
	}

	/*******
	 * === Price
	 */

	public function getPrice(bool $useCurrency = false): float
	{
		$price = (float) $this->price;

		$price = $useCurrency === true
			? $this->getOrder()->calculateCurrencyPrice($price)
			: $price;

		if (Config::load('roundOrderSales')) {
			$round = 0;
			if ((new DateTime())->format('Y-m-d') >= '2024-09-17') {
				$round = $this->getOrder()->currency->decimals ?? 0;
			}

			$price = round($price, $round);
		}

		return $price;
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$price = $this->getPrice($useCurrency);

		return round(OrderHelper::removeVat($price, $this->getVatRate()), 2);
	}

	public function getVat(bool $useCurrency = false): float
	{
		return $this->getPrice($useCurrency) - $this->getPriceWithoutVat($useCurrency);
	}

	public function getVatRate(): int
	{
		return (int) EshopOrdersConfig::load('discounts.vat');
	}

	/*******
	 * === Type
	 */

	public function getType(): string
	{
		return $this->type;
	}

	public function setType(?string $type): OrderDiscount
	{
		$this->type = $type;

		return $this;
	}

	/*******
	 * === Order
	 */

	public function getOrder(): Order
	{
		return $this->order;
	}

	public function setOrder(Order $order): self
	{
		$this->order = $order;

		return $this;
	}

	public function getSortPosition(): int
	{
		switch ($this->type) {
			case self::TYPE_PERCENT:
				return 1;
			case self::TYPE_FIX:
				return 2;
		}

		return 9999;
	}
}

