<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\FormValidateEvent;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\Products as FrontProducts;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderSpeditionHelper;
use EshopOrders\Model\OrderItems;
use EshopOrders\Model\Orders;
use Nette\Application\UI\Form;
use Nette\Utils\ArrayHash;

class OrderItemAddForm extends BaseControl
{
	public ?Order                  $order;
	protected Orders               $ordersService;
	protected OrderItems           $orderItemsService;
	protected Products             $productsService;
	protected FrontProducts        $frontProducts;
	protected OrderSpeditionHelper $orderSpeditionHelper;

	public function __construct(
		int                  $orderId,
		Orders               $orders,
		OrderItems           $items,
		Products             $products,
		FrontProducts        $frontProducts,
		OrderSpeditionHelper $orderSpeditionHelper
	)
	{
		$this->ordersService        = $orders;
		$this->orderItemsService    = $items;
		$this->productsService      = $products;
		$this->frontProducts        = $frontProducts;
		$this->order                = $this->ordersService->get($orderId);
		$this->orderSpeditionHelper = $orderSpeditionHelper;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	public function handleProductsLoad(): void
	{
		$output = [];
		$params = $this->getParameters();

		if (isset($params['site'])) {
			Sites::$currentIdentOverride = $params['site'];
		}

		$qb = $this->productsService->getEr()->createQueryBuilder('p')
			->select('p.id, p.code1, p.ean, p.price, vr.rate as vatRate, pt.name')
			->innerJoin('p.vatRate', 'vr')
			->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->andWhere('p.isDeleted = 0')
			->setParameter('lang', $this->translator->getLocale());

		if (Config::load('product.publishedByLang')) {
			$qb->andWhere('pt.isPublished = 1');
		} else {
			$qb->andWhere('p.isPublished = 1');
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$output[$row['id']] = [
				'id'      => (string) $row['id'],
				'name'    => (string) $row['name'],
				'code1'   => (string) $row['code1'],
				'ean'     => (string) $row['ean'],
				'price'   => (float) $row['price'],
				'vatRate' => (int) $row['vatRate'],
			];
		}

		$this->presenter->sendJson($output);
	}

	public function handleProductDetail(): void
	{
		$products = [];
		$params   = $this->getParameters();

		$userId  = $params['userId'] ? (int) $params['userId'] : null;
		$country = $params['country'] ?: null;

		$qb = $this->productsService->getEr()->createQueryBuilder('p')
			->select('p.id, p.code1, p.ean, p.price, vr.rate as vatRate, pt.name')
			->innerJoin('p.vatRate', 'vr')
			->leftJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->where('p.id = :id');

		if (Config::load('addManufacturerBeforeProductName', false)) {
			$qb->leftJoin('p.manufacturer', 'm')
				->addSelect('m.name as manufacturer');
		}

		$qb->setParameters([
			'id'   => $params['id'],
			'lang' => $this->order->lang,
		]);

		$arrProd = $qb->getQuery()->getOneOrNullResult();

		/** @var \EshopCatalog\FrontModule\Model\Dao\Product $daoClass */
		$daoClass                 = Config::load('product.daoClass');
		$products[$arrProd['id']] = (new $daoClass((int) $arrProd['id']))
			->setCode1((string) $arrProd['code1'])
			->setEan((string) $arrProd['ean'])
			->setPrice((float) $arrProd['price'])
			->setVatRate((int) $arrProd['vatRate']);
		$this->frontProducts->loadFresh($products, $userId, false, $country);

		/** @var \EshopCatalog\FrontModule\Model\Dao\Product $product */
		$product = $products[$arrProd['id']];
		$product->setName((string) trim($arrProd['manufacturer'] . ' ' . $arrProd['name']));

		$this->presenter->sendJson([
			'id'      => (string) $product->getId(),
			'name'    => (string) $product->name,
			'code1'   => (string) $product->getCode1(),
			'ean'     => (string) $product->getEan(),
			'price'   => $product->priceInBaseCurrency,
			'vatRate' => $product->getVatRate(),
		]);
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->getElementPrototype()->addClass('orderItemAddForm');

		$linkParams = [];
		try {
			$linkParams['site'] = $this->order->site->getIdent();

			$addrDeli = $this->order->getAddressDelivery();
			if ($addrDeli && $addrDeli->getCountry()) {
				$linkParams['country'] = $addrDeli->getCountry()->getId();
			}

			if ($this->order->getCustomer()) {
				$firstSignIn = $this->order->getCustomer()->getUser()->getFirstSignIn();

				if ($firstSignIn !== null) {
					$linkParams['userId'] = $this->order->getCustomer()->getUser()->getId();
				}
			}
		} catch (\Exception $e) {
		}

		$form->addHidden('productId')->setRequired('Vyberte prosím produkt');
		$form->addText('product', 'eshopOrders.default.productId')
			->setHtmlAttribute('data-autocomplete-name', 'orderItemAddForm-product')
			->setHtmlAttribute('data-autocomplete-url', $this->link('productsLoad!', $linkParams))
			->setHtmlAttribute('data-autocomplete-target', $form->getComponent('productId')->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setHtmlAttribute('data-detail-link', $this->link('productDetail!', $linkParams + ['id' => '__id__']))
			->setRequired();
		$form->addInteger('quantity', 'eshopOrders.default.quantity')->setRequired()->addRule(Form::MIN, 'Počet musí být alespoň %d', 1);
		$form->addText('code1', 'eshopOrders.itemsGrid.code1');
		$form->addText('price', 'eshopOrders.default.pricePiece')->setRequired()->addRule(Form::FLOAT, 'Cena musí být číslo');
		$form->addText('vatRate', 'eshopOrders.itemsGrid.vatRate')->setRequired()->addRule(Form::INTEGER, 'DPH musí být celé číslo');

		if (EshopOrdersConfig::load('orderForm.allowModifyNote')) {
			$form->addTextArea('note', 'eshopOrders.orderItem.note')
				->setHtmlAttribute('rows', 3);
		}

		$form->addHidden('productName', '');

		$form->addSaveCancelControl();
		$form->onSuccess[]  = [$this, 'formSuccess'];
		$form->onValidate[] = function(BaseForm $form, ArrayHash $values) {
			$event                      = new FormValidateEvent($form, $values,
				$this->presenter ? $this->template : null, $this->presenter ?: null);
			$event->custom['orderItem'] = null;
			$event->custom['order']     = $this->order;
			if ($this->presenter) {
				$event->presenter = $this->presenter;
			}

			$this->eventDispatcher->dispatch($event, self::class . '::formValidate');

			$this->redrawControl('formErrors');
		};

		$form->setDefaults([
			'quantity' => 1,
		]);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			/** @var Product|null $productRef */
			$productRef = $this->em->getReference(Product::class, $values->productId);

			$lang      = $this->translator->getLocale();
			$orderItem = new OrderItem($productRef, $this->order);
			$orderItem->setCode1($values->code1);
			$orderItem->addOrderItemText($lang);
			$orderItem->getOrderItemText($lang)->setName($values->productName);
			$orderItem->setQuantity($values->quantity);
			$orderItem->setPrice($values->price);
			$orderItem->setVatRate($values->vatRate);

			if (EshopOrdersConfig::load('orderForm.allowModifyNote')) {
				$orderItem->setMoreDataValue('note', $values->note);
			}

			$this->orderItemsService->addOrderItem($orderItem);

			if (EshopOrdersConfig::load('recalculateSpeditionPriceByItemsPriceVatRate')) {
				$this->orderSpeditionHelper->checkSpeditionVatRateByItemsPrice($this->order);
			}

			$this->presenter->flashMessageSuccess('eshopOrders.itemForm.added');

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}
}
