<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\SupplierToOrder;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Supplier;
use EshopOrders\AdminModule\Model\Orders;
use EshopOrders\Model\Entities\OrderItem;
use Nette\Utils\DateTime;
use Nette\Utils\Html;
use Nette\Utils\Json;

class SupplierGrid extends BaseControl
{
	protected int $supplierId;

	protected Orders    $orders;
	protected ?Supplier $supplier = null;

	public function __construct(
		int    $supplierId,
		Orders $orders
	)
	{
		$this->supplierId = $supplierId;
		$this->orders     = $orders;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function getSupplier(): Supplier
	{
		if (!$this->supplier) {
			/** @var Supplier $supplier */
			$supplier = $this->em->getRepository(Supplier::class)->find($this->supplierId);

			$this->supplier = $supplier;
		}

		return $this->supplier;
	}

	public function handleOrder(string $id): void
	{
		try {
			$items = Json::decode(base64_decode($id), Json::FORCE_ARRAY);

			foreach ($this->em->getRepository(OrderItem::class)->createQueryBuilder('oi')
				         ->where('oi.id IN (:items)')
				         ->setParameter('items', $items)
				         ->getQuery()->getResult() as $item) {
				/** @var OrderItem $item */
				$item->setMoreDataValue('supplierOrdered', (new DateTime())->format('Y-m-d H:i:s'));

				$this->em->persist($item);
			}

			$this->em->flush();

			$this['grid']->reload();
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redrawControl('flashes');
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$data = [];

		foreach ($this->em->getRepository(OrderItem::class)->createQueryBuilder('oi')
			         ->select('oi.id, o.id as orderId, p.id as productId, p.code1, oit.name, oi.quantity, oi.moreData')
			         ->innerJoin('oi.orderItemTexts', 'oit', Join::WITH, 'oit.lang = :lang')
			         ->innerJoin('oi.order', 'o', Join::WITH, ' o.id IN (:orders)')
			         ->innerJoin('oi.product', 'p')
			         ->innerJoin('p.suppliers', 'ps', Join::WITH, 'ps.supplier = :supplier')
			         ->setParameters([
				         'orders'   => $this->orders->getOrdersNotFinishedOrCancelled(),
				         'supplier' => $this->supplierId,
				         'lang'     => $this->translator->getLocale(),
			         ])
			         ->orderBy('oi.order', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			$moreData  = $row['moreData'] ?: [];
			$isOrdered = ($moreData['supplierOrdered'] ?: null) !== null;

			$key = md5($row['productId'] . '-' . ($isOrdered ? 1 : 0) . '-' . $moreData['note']);

			if (!isset($data[$key])) {
				$orderDate = null;
				if ($moreData['supplierOrdered']) {
					$orderDate = DateTime::createFromFormat('Y-m-d H:i:s', $moreData['supplierOrdered']);

					if ($orderDate) {
						$orderDate = $orderDate->format('j. n. Y H:i');
					} else {
						$orderDate = null;
					}
				}

				$data[$key] = [
					'key'       => $key,
					'id'        => $row['productId'],
					'name'      => $row['name'],
					'quantity'  => $row['quantity'],
					'orders'    => [$row['orderId']],
					'note'      => $moreData['note'] ?? null,
					'code1'     => $row['code1'],
					'items'     => [$row['id']],
					'isOrdered' => $isOrdered,
					'orderDate' => $orderDate,
				];
			} else {
				$data[$key]['quantity'] += $row['quantity'];
				$data[$key]['orders'][] = $row['orderId'];
				$data[$key]['items'][]  = $row['id'];
			}
		}

		foreach ($data as $k => $v) {
			$data[$k]['order']    = array_unique($data[$k]['orders']);
			$data[$k]['items']    = array_unique($data[$k]['items']);
			$data[$k]['itemsKey'] = base64_encode(Json::encode($data[$k]['items']));
		}

		$grid->setPrimaryKey('key');
		$grid->setDataSource($data);

		$grid->addColumnText('code1', 'eshopOrders.supplierToOrder.supplierGrid.code1')
			->setRenderer(function(array $row): Html {
				$wrap = Html::el($row['isOrdered'] ? 'del' : 'span');
				$wrap->setText($row['code1']);

				return $wrap;
			});
		$grid->addColumnText('product', 'eshopOrders.supplierToOrder.supplierGrid.product')
			->setRenderer(function(array $row): Html {
				$wrap = Html::el();
				$wrap->addHtml(Html::el('a', ['target' => '_blank'])
					->href($this->presenter->link(':EshopCatalog:Admin:Products:edit', ['id' => $row['id']]))
					->setText($row['name']));

				if ($row['isOrdered']) {
					$wrap->addHtml(Html::el('span')->setText(' - (' . $row['orderDate'] . ')'));
				}

				$wrap->addHtml(Html::el('div')->setText($row['note']));

				return $wrap;
			});
		$grid->addColumnText('orders', 'eshopOrders.supplierToOrder.supplierGrid.orders')
			->setRenderer(function(array $row): Html {
				$wrap = Html::el($row['isOrdered'] ? 'del' : 'span');

				$orders = array_unique($row['orders']);
				$lastK  = array_key_last($orders);

				foreach ($orders as $k => $orderId) {
					$wrap->addHtml(Html::el('a', ['target' => '_blank'])
						->href($this->presenter->link(':EshopOrders:Admin:Default:editOrder', ['id' => $orderId]))
						->setText($orderId));

					if ($k !== $lastK) {
						$wrap->addHtml(', ');
					}
				}

				return $wrap;
			});
		$grid->addColumnNumber('quantity', 'eshopOrders.supplierToOrder.supplierGrid.quantity')
			->setRenderer(function(array $row): Html {
				$wrap = Html::el($row['isOrdered'] ? 'del' : 'span');
				$wrap->setText($row['quantity']);

				return $wrap;
			});

		$grid->addAction('order', '', 'order!', ['id' => 'itemsKey'])
			->setIcon('fas fa-check')
			->setBsType('success')
			->addClass('ajax')
			->setRenderCondition(function(array $row): bool {
				return !$row['isOrdered'];
			});

		$grid->getColumn('quantity')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('code1')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('orders')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}
}
