<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestBody;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use EshopOrders\ApiModule\Api\V1\BodyEntity\AttachFile;
use EshopOrders\ApiModule\Api\V1\Exceptions\ErrorException;
use EshopOrders\ApiModule\Api\V1\Model\Orders;
use EshopOrders\Model\Event\OrderPayloadReturnEvent;
use Exception;

/**
 * @Path("/orders")
 */
class OrdersController extends BaseController
{
	protected EventDispatcher        $eventDispatcher;
	protected EntityManagerDecorator $em;
	protected Orders                 $orders;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		EntityManagerDecorator $em,
		Orders                 $orders
	)
	{
		$this->em              = $em;
		$this->eventDispatcher = $eventDispatcher;
		$this->orders          = $orders;
	}

	/**
	 * @Path("/")
	 * @Method("GET")
	 */
	public function index(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getStatusesOverview()));
	}

	/**
	 * @Path("/{id}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 */
	public function getDetail(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		try {
			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from(['order' => $this->orders->getDetail($request->getParameter('id'))]));
		} catch (ErrorException $ex) {
			return $this->sendError($response, 'orderNotFound');
		}
	}

	/**
	 * @Path("/status/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getIdByStatus($request->getParameter('status'))));
	}

	/**
	 * @Path("/status-not/{status}")
	 * @Method("GET")
	 * @RequestParameters({
	 *      @RequestParameter(name="status", type="string", required=true),
	 * })
	 */
	public function getByStatusNot(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($this->orders->getIdByStatusNot($request->getParameter('status'))));
	}

	/**
	 * @Path("/{id}/status")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\OrderStatus")
	 */
	public function setOrderStatus(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->setOrderStatus(
					$request->getParameter('id'),
					$request->getEntity(),
				),
			]));
	}

	/**
	 * @Path("/{id}/payment")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Payment")
	 */
	public function changePayment(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->changePayment(
						$request->getParameter('id'),
						$request->getEntity(),
					) !== null,
			]));
	}

	/**
	 * @Path("/{id}/spedition")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="id", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\Spedition")
	 */
	public function changeSpedition(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->changeSpedition(
						$request->getParameter('id'),
						$request->getEntity(),
					) !== null,
			]));
	}

	/**
	 * @Path("/{id}/attached-files-list")
	 * @Method("GET")
	 * @RequestParameters({
	 *       @RequestParameter(name="id", type="int", required=true),
	 *  })
	 */
	public function getAttachedFilesList(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->orders->getAttachedFiles((int) $request->getParameter('id')),
			]));
	}

	/**
	 * @Path("/{id}/attach-file")
	 * @Method("POST")
	 * @RequestParameters({
	 *       @RequestParameter(name="id", type="int", required=true),
	 *  })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\AttachFile")
	 */
	public function attachFile(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		/** @var AttachFile $dto */
		$dto = $request->getEntity();

		try {
			$this->orders->attachFile((int) $request->getParameter('id'), $dto->fileName, $dto->fileContent);

			return $response->withStatus(ApiResponse::S200_OK)
				->withEntity(ArrayEntity::from([
					'result' => $this->orders->getAttachedFiles((int) $request->getParameter('id')),
				]));
		} catch (Exception $e) {
			return $response->withStatus(ApiResponse::S400_BAD_REQUEST)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
			]));
		}
	}

	/**
	 * @Path("/{id}/remove-attached-file")
	 * @Method("DELETE")
	 * @RequestParameters({
	 *       @RequestParameter(name="id", type="int", required=true),
	 *  })
	 */
	public function deleteFile(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$fileName = (string) $request->getBody() ? $request->getJsonBody()['fileName'] : null;

		if (!$fileName) {
			return $response->withStatus(ApiResponse::S400_BAD_REQUEST)->withEntity(ArrayEntity::from([
				'error' => 'fileName is required',
			]));
		}

		try {
			$this->orders->deleteFile((int) $request->getParameter('id'), $fileName);

			return $response->withStatus(ApiResponse::S200_OK)
				->withEntity(ArrayEntity::from([
					'result' => $this->orders->getAttachedFiles((int) $request->getParameter('id')),
				]));
		} catch (Exception $e) {
			return $response->withStatus(ApiResponse::S400_BAD_REQUEST)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
			]));
		}
	}

	/**
	 * @Path("/create")
	 * @Method("POST")
	 */
	public function create(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		$data = $request->getJsonBody();

		try {
			$order  = $this->orders->createForCheckout($data);
			$result = ['success' => $order];

			$orderPayloadReturnEvent = new OrderPayloadReturnEvent($result, $order);
			$this->eventDispatcher->dispatch($orderPayloadReturnEvent, 'eshopOrders.api.orderPayloadReturnEvent');

			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from($orderPayloadReturnEvent->data));
		} catch (Exception $e) {
			return $response->withStatus(ApiResponse::S200_OK)->withEntity(ArrayEntity::from([
				'error' => $e->getMessage(),
				'data'  => $data,
			]));
		}
	}
}
