<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Controllers;

use Apitte\Core\Annotation\Controller\Method;
use Apitte\Core\Annotation\Controller\Path;
use Apitte\Core\Annotation\Controller\RequestBody;
use Apitte\Core\Annotation\Controller\RequestParameter;
use Apitte\Core\Annotation\Controller\RequestParameters;
use Apitte\Core\Http\ApiRequest;
use Apitte\Core\Http\ApiResponse;
use Apitte\Negotiation\Http\ArrayEntity;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use EshopOrders\ApiModule\Api\V1\BodyEntity\OrderItem;
use EshopOrders\ApiModule\Api\V1\Model\OrdersItem;

/**
 * @Path("/orders-item")
 */
class OrdersItemController extends BaseController
{
	protected EventDispatcher        $eventDispatcher;
	protected EntityManagerDecorator $em;
	protected OrdersItem             $ordersItem;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		EntityManagerDecorator $em,
		OrdersItem             $ordersItem
	)
	{
		$this->em              = $em;
		$this->eventDispatcher = $eventDispatcher;
		$this->ordersItem      = $ordersItem;
	}

	/**
	 * @Path("/{orderId}/{itemId}/price")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="orderId", type="int", required=true),
	 *      @RequestParameter(name="itemId", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\OrderItem")
	 */
	public function changePrice(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		/** @var OrderItem $entity */
		$entity = $request->getEntity();

		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->ordersItem->changePrice(
					$request->getParameter('orderId'),
					$request->getParameter('itemId'),
					$entity->price,
				),
			]));
	}

	/**
	 * @Path("/{orderId}/{itemId}/quantity")
	 * @Method("POST")
	 * @RequestParameters({
	 *      @RequestParameter(name="orderId", type="int", required=true),
	 *      @RequestParameter(name="itemId", type="int", required=true),
	 * })
	 * @RequestBody(entity="EshopOrders\ApiModule\Api\V1\BodyEntity\OrderItem")
	 */
	public function changeQuantity(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		/** @var OrderItem $entity */
		$entity = $request->getEntity();

		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->ordersItem->changeQuantity(
					$request->getParameter('orderId'),
					$request->getParameter('itemId'),
					$entity->quantity ?: 1,
				),
			]));
	}

	/**
	 * @Path("/{orderId}/{itemId}")
	 * @Method("DELETE")
	 * @RequestParameters({
	 *      @RequestParameter(name="orderId", type="int", required=true),
	 *      @RequestParameter(name="itemId", type="int", required=true),
	 * })
	 */
	public function removeItem(ApiRequest $request, ApiResponse $response): ApiResponse
	{
		return $response->withStatus(ApiResponse::S200_OK)
			->withEntity(ArrayEntity::from([
				'result' => $this->ordersItem->removeItem(
					$request->getParameter('orderId'),
					$request->getParameter('itemId'),
				),
			]));
	}
}
