<?php declare(strict_types = 1);

namespace EshopOrders\ApiModule\Api\V1\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\Country;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\PaymentSpedition;
use Nette\Utils\Strings;

class PaymentSpeditions
{
	protected EntityManagerDecorator $em;
	protected Sites $sitesService;
	protected Translator $translator;

	public function __construct(EntityManagerDecorator $em, Sites $sitesService, Translator $translator)
	{
		$this->em = $em;
		$this->sitesService = $sitesService;
		$this->translator = $translator;
	}

	/**
	 * @return Dao\PaymentSpedition[]
	 */
	public function getAll(): array
	{
		$result = [];
		$qb = $this->em->getRepository(PaymentSpedition::class)->createQueryBuilder('ps')
				   ->select('ps.id as id, p.id as payment, s.id as spedition, GROUP_CONCAT(c.id) as countries, ps.freeFromDate, ps.freeToDate, ps.currency')
				   ->andWhere('ps.isPublished = :isPublished')
				   ->innerJoin('ps.countries', 'c')
				   ->innerJoin('ps.payment', 'p')
				   ->innerJoin('ps.spedition', 's')
				   ->addOrderBy('s.position', 'ASC')
				   ->addOrderBy('p.position', 'ASC')
				   ->setParameters([
					   'isPublished' => 1,
				   ])
				   ->groupBy('ps.id');

		if (count($this->sitesService->getSites()) > 1) {
			$qb->innerJoin('ps.sites', 'sites', Join::WITH, 'sites.ident = :site')
			   ->setParameter('site', $this->sitesService->getCurrentSite()->getIdent());
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$dao = new Dao\PaymentSpedition($row['id'], $row['payment'], $row['spedition']);
			if ($row['countries']) {
				$dao->countries = Strings::contains($row['countries'], ',') ? array_map('trim', explode(',', $row['countries'])) : [$row['countries']];
			} else {
				$dao->countries = [];
			}

			$dao->currency = $row['currency'];
			$dao->freeFromDate = $row['freeFromDate'];
			$dao->freeToDate = $row['freeToDate'];

			$result[$row['id']] = $dao;
		}

		return $result;
	}

	/**
	 * @return array<string, string>
	 */
	public function getCountries(): array
	{
		$countries = [];
		foreach ($this->getAll() as $ps) {
			foreach ($ps->countries as $country) {
				$countries[$country] = $country;
			}
		}

		if (!$countries) {
			return [];
		}

		$qb = $this->em->getRepository(Country::class)->createQueryBuilder('c', 'c.id');
		$qb->select('c.id, ct.name')
		   ->innerJoin('c.texts', 'ct', Join::WITH, 'ct.lang = :lang')
		   ->where($qb->expr()->in('c.id', $countries))
		   ->orderBy('c.position')
		   ->setParameter('lang', $this->translator->getLocale());

		$result = [];
		foreach ($qb->getQuery()->getArrayResult() as $r) {
			$result[$r['id']] = $r['name'];
		}

		return $result;

	}

}