<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use EshopOrders\FrontModule\Model\Dao\AddOrUpdateItem;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\PaymentSpeditions;

class CartFacade
{
	protected Carts             $carts;
	protected Speditions        $speditions;
	protected Payments          $payments;
	protected PaymentSpeditions $paymentSpeditions;
	protected CartHelper        $cartHelper;
	protected EventDispatcher   $eventDispatcher;

	public function __construct(
		Carts             $carts,
		Speditions        $speditions,
		Payments          $payments,
		PaymentSpeditions $paymentSpeditions,
		CartHelper        $cartHelper,
		EventDispatcher   $eventDispatcher
	)
	{
		$this->carts             = $carts;
		$this->speditions        = $speditions;
		$this->payments          = $payments;
		$this->paymentSpeditions = $paymentSpeditions;
		$this->cartHelper        = $cartHelper;
		$this->eventDispatcher   = $eventDispatcher;
	}

	public function getCart(): Cart
	{
		$cart = $this->carts->getCurrentCart();

		$this->paymentSpeditions->cart = $cart;

		$formData = $this->getOrderFormData();
		if ($formData) {
			if ($formData['spedition'] && $v = $this->speditions->get($formData['spedition'])) {
				$cart->setSpedition($v);
			}
			if ($formData['payment'] && $v = $this->payments->get($formData['payment'])) {
				$cart->setPayment($v);
			}

			if ($cart->spedition && $cart->payment) {
				$cart->payment->vat = $cart->spedition->vat;
			}
		}

		if (EshopOrdersConfig::load('allowFreeFrom') && $cart->spedition && $cart->spedition->price > 0) {
			$freeCombinations = [];
			foreach ($this->paymentSpeditions->getAllPublishedByCountry($formData ? $formData['speditionCountry'] : null) as $ps) {
				if ($ps->isFreeCombination()) {
					$freeCombinations[] = $ps;
				}
			}

			$cart->paymentSpeditionFreeCombinations = $freeCombinations;
		}

		return $cart;
	}

	public function getCartsService(): Carts { return $this->carts; }

	public function getOrderFormData(): array { return $this->cartHelper->getOrderFormData(); }

	public function countFreeSpedition(): ?array { return $this->cartHelper->countFreeSpedition(); }

	public function countFreeSpeditionMilestones(): ?array { return $this->cartHelper->countFreeSpeditionMilestones(); }

	public function addOrUpdateItem(AddOrUpdateItem $item): ?CartItem
	{
		$cart = $this->carts->getCurrentCart();

		$getCartItem = static function(Cart $cart, ?int $productId, $itemId): ?CartItem {
			if ($productId) {
				$cartItem = $cart->getCartItemByProductId($productId);
			} else if ($itemId) {
				$cartItem = $cart->getCartItemByIdent((string) $itemId);
			}

			return $cartItem ?? null;
		};

		$cartItem = $getCartItem($cart, $item->productId, $item->itemId);

		if ($cartItem) {
			$this->eventDispatcher->dispatch(new UpdatedCartItemEvent(
				$item->itemId,
				$item->quantity,
				$cartItem->quantity,
				$item->moreData,
				$cartItem->productId,
			), 'eshopOrders.cartUpdateItem');
		} else {
			$addedItem           = new AddedCartItem();
			$addedItem->quantity = $item->quantity ?: 1;
			$addedItem->moreData = $item->moreData;
			$addedItem->name     = $item->name;
			$addedItem->price    = $item->price;
			$addedItem->itemId   = $item->itemId;
			$addedItem->image    = $item->image;

			if ($item->productId) {
				$addedItem->productId = $item->productId;
			}

			$this->eventDispatcher->dispatch(new AddedCartItemEvent($addedItem), 'eshopOrders.cartAddItem');
		}

		return $getCartItem($this->getCartsService()->getCurrentCart(), $item->productId, $item->itemId);
	}

	public function mergeCartToUserCart(int $userId): void
	{
		$userCartRaw = $this->getCartsService()->findUserCart($userId);
		if (!$userCartRaw) {
			return;
		}

		$oldCart = $this->getCartsService()->getCurrentCart();
		if (!$oldCart->id) {
			$this->getCartsService()->assignCart($userCartRaw->getId());

			return;
		}

		// Pokud je puvodni kosik prazdny, tak smazat a priradit uzivateluv
		if (empty($oldCart->cartItems)) {
			$this->getCartsService()->deleteCurrentCart();
			$this->getCartsService()->assignCart($userCartRaw->getId());

			return;
		}

		// Jinak pridat polozky z puvodniho kosiku do uzivatelova
		$userCart = $this->getCartsService()->assignCart($userCartRaw->getId());

		$getCartItem = static function(Cart $cart, ?int $productId, $itemId): ?CartItem {
			if ($productId) {
				$cartItem = $cart->getCartItemByProductId($productId);
			} else if ($itemId) {
				$cartItem = $cart->getCartItemByIdent((string) $itemId);
			}

			return $cartItem ?? null;
		};

		foreach ($oldCart->getCartItems() as $cartItem) {
			$existCartItem = $getCartItem($userCart, $cartItem->productId, $cartItem->id);

			if (!$existCartItem) {
				$addedItem           = new AddedCartItem();
				$addedItem->quantity = $cartItem->quantity ?: 1;
				$addedItem->moreData = $cartItem->getData();
				$addedItem->name     = $cartItem->title;
				$addedItem->price    = $cartItem->price;
				$addedItem->itemId   = (string) $cartItem->id;
				$addedItem->image    = $cartItem->image ? $cartItem->getImage()->getFilePath() : null;

				if ($cartItem->productId) {
					$addedItem->productId = $cartItem->productId;
				}

				$this->eventDispatcher->dispatch(new AddedCartItemEvent($addedItem), 'eshopOrders.cartAddItem');
			}
		}

		// Stary smazat
		$this->getCartsService()->deleteCartById($oldCart->id);
	}
}
