<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Contributte\Psr7\Psr7UploadedFile;
use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Helpers\BaseService;
use Core\Model\Helpers\Strings;
use Core\Model\Templating\Filters\Price as PriceFilter;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\PaymentSpeditions;
use EshopSales\FrontModule\Model\OrderSales;
use Nette\DI\Container;
use Nette\Http\FileUpload;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\SmartObject;
use Nette\Utils\FileSystem;
use Nette\Utils\Html;
use Users\Model\Security\User;

class CartHelper extends BaseService
{
	use SmartObject;

	public const SESSION_KEY = 'eshopOrdersOrderForm';

	protected SessionSection    $sessionSection;
	protected Carts             $cartsService;
	protected Speditions        $speditionsService;
	protected Customers         $customers;
	protected Container         $container;
	protected User              $user;
	protected PaymentSpeditions $paymentSpedition;
	protected Translator        $translator;
	protected AppState          $appState;
	protected PriceFilter       $priceFilter;
	protected ?OrderSales       $orderSales = null;

	public function __construct(
		Carts             $carts,
		Speditions        $speditions,
		Customers         $customers,
		User              $user,
		Container         $container,
		Session           $session,
		PaymentSpeditions $paymentSpeditions,
		Translator        $translator,
		AppState          $appState,
		PriceFilter       $priceFilter
	)
	{
		$this->cartsService      = $carts;
		$this->speditionsService = $speditions;
		$this->customers         = $customers;
		$this->user              = $user;
		$this->container         = $container;
		$this->sessionSection    = $session->getSection(self::SESSION_KEY);
		$this->paymentSpedition  = $paymentSpeditions;
		$this->translator        = $translator;
		$this->appState          = $appState;
		$this->priceFilter       = $priceFilter;
	}

	// --------------- Doprava zdarma ------------

	public function countFreeSpedition(): array
	{
		$freeSpeditionPrice = $remainingPrice = null;

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$cart             = $this->cartsService->getCurrentCart();
			$speditionCountry = $this->appState->getCountry() ?? 'CZ';

			$firstFreeSpedition = $this->paymentSpedition->getFirstFreeSpedition($speditionCountry);
			if ($firstFreeSpedition) {
				$freeSpeditionPrice = $firstFreeSpedition->getFreeFrom();

				$cart      = $this->cartsService->getCurrentCart();
				$cartPrice = $cart->getPriceForFreeSpedition();

				if ($cartPrice > $freeSpeditionPrice) {
					$remainingPrice = 0;
				} else {
					$remainingPrice = $freeSpeditionPrice - $cartPrice;
				}
			}

			foreach ($cart->getCartItems() as $item) {
				if ($item->freeDelivery === true) {
					$freeSpeditionPrice = $item->getPrice(); //ne 0 -> to by znamenalo ze neni zdarma nikdy
					$remainingPrice     = 0;
				} else {
					if (!$firstFreeSpedition) {
						$freeSpeditionPrice = null;
						$remainingPrice     = null;
						break;
					}
				}
			}
		}

		return ['minPrice' => $freeSpeditionPrice, 'remaining' => $remainingPrice];
	}

	public function countFreeSpeditionMilestones(): array
	{
		$result = [];

		if (!EshopOrdersConfig::load('allowFreeFrom')) {
			return $result;
		}

		$cart             = $this->cartsService->getCurrentCart();
		$cartPrice        = $cart->getPriceForFreeSpedition();
		$speditionCountry = $this->getOrderFormData()['speditionCountry'] ?? $this->appState->getCountry() ?: 'CZ';
		$speditions       = $this->paymentSpedition->getFirstFreeSpeditionByType($speditionCountry);

		foreach (['storePickup', 'pickup', 'address'] as $k) {
			$firstFreeSpedition = $speditions[$k];
			$freeSpeditionPrice = $remainingPrice = null;

			if ($firstFreeSpedition) {
				$freeSpeditionPrice = $firstFreeSpedition->getFreeFrom();

				if ($cartPrice > $freeSpeditionPrice) {
					$remainingPrice = 0;
				} else {
					$remainingPrice = $freeSpeditionPrice - $cartPrice;
				}
			}

			foreach ($cart->getCartItems() as $item) {
				if ($item->freeDelivery === true) {
					$freeSpeditionPrice = $item->getPrice(); //ne 0 -> to by znamenalo ze neni zdarma nikdy
					$remainingPrice     = 0;
					break;
				}
			}

			$result[$k] = ['minPrice' => $freeSpeditionPrice, 'remaining' => $remainingPrice];
		}

		return $result;
	}

	public function getOrderFormData(): array { return $this->sessionSection->orderFormData ?: []; }

	public function setOrderSales(OrderSales $orderSales): void
	{
		$this->orderSales = $orderSales;
	}

	public function getItemsCountFormatted(int $count): Html
	{
		if (EshopOrdersConfig::load('cartPreview.style') === 2) {
			return Html::el()->setText($count);
		}

		if ($count > 0) {
			return Html::el()->setHtml($count .
				' <span class="cart-preview__items-text">' . $this->translator->translate('eshopCatalogFront.cartAddForm.items') . '</span>');
		}

		return Html::el()->setHtml('<span>0 ' . $this->translator->translate('eshopCatalogFront.cartAddForm.items') . '</span>');
	}

	public function getPriceFormatted(float $priceWithVat, float $priceWithoutVat): Html
	{
		$price = EshopOrdersConfig::load('cartPreview.showPriceWithoutVat')
			? $priceWithoutVat
			: $priceWithVat;

		return Html::el()->setHtml($this->priceFilter->format($price));
	}

	public function uploadFiles(CartItem $cartItem, array $files): array
	{
		$result     = [];
		$folderName = $cartItem->id;
		$fileDir    = DS . 'eshoporders-cart-uploads' . DS . $folderName . DS;
		FileSystem::createDir(UPLOADS_DIR . $fileDir);

		foreach ($files as $file) {
			if ($file instanceof Psr7UploadedFile) {
				$name     = Strings::webalize($file->getClientFilename(), '.', false);
				$name     = str_replace(['-.', '.-'], '.', $name);
				$filePath = $fileDir . $name;

				$file->moveTo(UPLOADS_DIR . $filePath);
			} else if ($file instanceof FileUpload) {
				if ($file->isOk()) {
					$filePath = $fileDir . $file->getSanitizedName();

					$file->move(UPLOADS_DIR . $filePath);
				}
			} else if (is_string($file)) {
				$tmp = explode('/uploads/', $file, 2);
				FileSystem::copy(UPLOADS_DIR . DS . $tmp[1], UPLOADS_DIR . $fileDir . basename($file));
			}
		}

		return $result;
	}

	public static function canAddToCart(Product $product): bool
	{
		if (!$product->getAvailability() || !$product->getAvailability()->canAddToCart()) {
			return false;
		}

		$quantity = $product->getQuantity() + $product->getQuantityExternal();

		return !EshopCatalogConfig::load('pseudoWarehouse')
			|| $quantity > 0
			|| (int) $product->unlimitedQuantity === 1
			|| (
				EshopCatalogConfig::load('allowWarehouseOverdraft')
				&& $product->getAvailability()->warehouseOverdraft
			);
	}
}
