<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\EshopOrdersConfig;
use Users\Model\Entities\User;
use Users\Model\Roles;
use Users\Model\Users;

/**
 * @method Customer|object|null getReference($id)
 * @method Customer[]|null getAll()
 * @method Customer|null get($id)
 */
class Customers extends BaseEntityService
{
	public const CUSTOMER = 'customer';

	protected $entityClass = Customer::class;

	protected Roles $rolesService;
	protected Users $usersService;

	protected array $cCustomers = [];

	public function __construct(
		Users $users,
		Roles $roles
	)
	{
		$this->usersService = $users;
		$this->rolesService = $roles;
	}

	/**
	 * @param User|int $user
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByUser($user): ?Customer
	{
		if ($user instanceof User) {
			$userId = $user->getId();
		} else {
			$userId = (int) $user;
		}

		if (!array_key_exists($userId, $this->cCustomers)) {
			$qb = $this->getEr()->createQueryBuilder('c', 'c.id')
				->addSelect('addrAv')
				->leftJoin('c.addressesAvailable', 'addrAv')
				->andWhere('c.user = :user')->setParameter('user', $userId);

			if (EshopOrdersConfig::load('enableDisablingPaymentSpeditionForCustomer', false)) {
				$qb->leftJoin('c.disabledPayments', 'disP')
					->leftJoin('c.addressDelivery', 'deli')
					->leftJoin('c.addressInvoice', 'inv')
					->leftJoin('c.disabledSpeditions', 'disS')
					->leftJoin('c.groupCustomers', 'gc')
					->leftJoin('gc.params', 'gcp')
					->addSelect('disP, disS, deli, inv, gc, gcp');
			}

			$this->cCustomers[$userId] = $qb->getQuery()->getOneOrNullResult();
		}

		return $this->cCustomers[$userId] ?? null;
	}

	/**
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getByUserId(?int $userId): ?Customer
	{
		return $userId ? $this->getByUser($userId) : null;
	}

	public function getOrCreateCustomer(?int $userId, ?string $email = null, string $firstName = '', ?string $lastName = '', ?string $phone = null): Customer
	{
		//user
		$user = $this->usersService->getOrCreateUser($userId, $email, $firstName, $lastName);

		//ma user prava "customer"? Pokud ne, tak pridat prava
		if (!$user->isInRoleByIdent(self::CUSTOMER)) {
			$role = $this->rolesService->getByIdent(self::CUSTOMER);
			if ($role) {
				$user->addRole($role);
				$this->em->persist($user)->flush();
			}
		}

		//customer
		$customer = $this->getByUser($user);
		if (!$customer) {
			$customer = new Customer($user, $phone);
			$this->em->persist($customer);
			$this->em->flush();

			$this->cCustomers = [];
			$customer         = $this->getByUser($user);
		}

		return $customer;
	}

	public function checkExistEmail(string $email): bool
	{
		return (bool) $this->em->getConnection()->fetchOne('SELECT id FROM user WHERE email = :email LIMIT 1', ['email' => $email]);
	}

	public function checkExistIco(string $ico): bool
	{
		if ($ico === '') {
			return false;
		}

		return (bool) $this->em->getConnection()->fetchOne('SELECT c.id FROM eshop_orders__customer c 
    			INNER JOIN eshop_orders__customer_address ca ON ca.id = c.address_invoice AND ca.id_number = :ico
                LIMIT 1', ['ico' => $ico]);
	}
}
