<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use Core\Model\Application\AppState;
use Nette\SmartObject;

class BasePS
{
	use SmartObject;

	public int                 $id;
	public ?Cart               $cart                   = null;
	public ?string             $name                   = null;
	public ?string             $ident                  = null;
	public ?string             $text                   = null;
	public int                 $isPublished            = 0;
	public int                 $position               = 0;
	public float               $price                  = 0;
	public int                 $vat                    = 0;
	public ?float              $priceInBaseCurrency    = null;
	public ?float              $freeFrom               = null;
	public ?float              $freeFromInBaseCurrency = null;
	public int                 $availableFrom          = 0;
	public ?int                $availableTo            = null;
	public ?string             $image                  = null;
	public ?\DateTimeInterface $freeFromDate           = null;
	public ?\DateTimeInterface $freeToDate             = null;
	public ?string             $country                = null;
	public array               $extras                 = [];

	public function getId(): int { return $this->id; }

	public function setId(int $id): self
	{
		$this->id = $id;

		return $this;
	}

	public function getIdent(): ?string { return $this->ident; }

	public function setIdent(?string $ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getText(): ?string { return $this->text; }

	public function setText(string $text): self
	{
		$this->text = $text;

		return $this;
	}

	public function isPublished(): bool { return (bool) $this->isPublished; }

	public function setPublished(int $isPublished): self
	{
		$this->isPublished = $isPublished;

		return $this;
	}

	public function getPosition(): int { return $this->position; }

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function getPrice(): float { return $this->price; }

	public function getPriceWithoutVat(): float { return round(($this->price / (100 + $this->vat)) * 100, AppState::getState('eshopOrdersCartDecimals') ?? 0); }

	public function getPriceInBaseCurrency(): float { return $this->priceInBaseCurrency; }

	public function getPriceInBaseCurrencyWithoutVat(): float { return round(($this->priceInBaseCurrency / (100 + $this->vat)) * 100, AppState::getState('eshopOrdersCartDecimals') ?? 0); }

	public function setPrice(float $price): self
	{
		$this->price = $price;

		return $this;
	}

	public function setCart(?Cart $cart = null): self
	{
		$this->cart = &$cart;

		return $this;
	}

	/**
	 * vrati cenu podle kosiku:
	 *  - podle celkove ceny objednavky muze byt doprava zdarma
	 *  - kdyz je v kosiku produkt se stitkem "doprava zdarma"
	 *
	 * @param Cart $cart
	 * @param bool $returnInBaseCurrency
	 *
	 * @return float
	 */
	public function getPriceActual(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		$cart  = $cart ?: $this->cart;
		$price = $this->prepareActualPrice($cart, $returnInBaseCurrency);

		if ($cart->payment && $cart->spedition) {
			foreach ($cart->paymentSpeditionFreeCombinations as $ps) {
				if ($ps->getPayment()->getId() == $cart->payment->getId() && $ps->getSpedition()->getId() == $cart->spedition->getId()) {
					return 0;
				}
			}
		}

		return $returnInBaseCurrency
			? $price
			: round($price, AppState::getState('eshopOrdersCartDecimals') ?? 0);
	}

	public function getPriceActualWithoutVat(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		return round(($this->getPriceActual($cart, $returnInBaseCurrency) / (100 + $this->vat)) * 100, AppState::getState('eshopOrdersCartDecimals') ?? 0);
	}

	public function getSpeditionChoicePrice(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		$cart = $cart ?: $this->cart;

		return $this->prepareActualPrice($cart, $returnInBaseCurrency);
	}

	protected function prepareActualPrice(?Cart $cart = null, bool $returnInBaseCurrency = false): float
	{
		$cart     = $cart ?: $this->cart;
		$freeFrom = $this->getFreeFrom();

		if ($freeFrom !== null && $freeFrom <= $cart->getCartItemsPrice()) {
			return 0;
		}

		// Pouze pro dopravu, ne platbu
		if (get_class($this) !== Payment::class) {
			$freeDelivery = true;

			foreach ($cart->getCartItems() as $item) {
				if ($item->freeDelivery !== true) {
					$freeDelivery = false;
				}
			}

			if ($freeDelivery) {
				return 0;
			}
		}

		return $returnInBaseCurrency ? $this->getPriceInBaseCurrency() : $this->getPrice();
	}

	public function getFreeFrom(): ?float { return $this->freeFrom; }

	public function getFreeFromInBaseCurrency(): float { return $this->freeFromInBaseCurrency; }

	public function setFreeFrom(?float $freeFrom): self
	{
		$this->freeFrom = $freeFrom;

		return $this;
	}

	public function getAvailableFrom(): int { return $this->availableFrom ?: 0; }

	public function setAvailableFrom(int $availableFrom): self
	{
		$this->availableFrom = $availableFrom;

		return $this;
	}

	public function getAvailableTo(): int { return $this->availableTo ?: 999999999; }

	public function setAvailableTo(int $availableTo): self
	{
		$this->availableTo = $availableTo;

		return $this;
	}

	public function getImage(): ?string { return $this->image; }

	public function setImage(string $image): self
	{
		$this->image = $image;

		return $this;
	}
}
