<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use Core\Model\Helpers\Arrays;
use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Components\Customer\AddressesList;
use EshopOrders\FrontModule\Components\Customer\IAddressesListFactory;
use EshopOrders\FrontModule\Components\Customer\IOrdersListFactory;
use EshopOrders\FrontModule\Components\Customer\IProfileFormFactory;
use EshopOrders\FrontModule\Components\Customer\OrdersList;
use EshopOrders\FrontModule\Components\Customer\ProfileForm;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Orders;
use EshopOrders\FrontModule\Model\ProductsAssignedToCustomers;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Nette\Application\BadRequestException;
use Nette\Application\Responses\FileResponse;
use Nette\Application\UI\Multiplier;
use Users\FrontModule\Components\IOAuthUserConnectFactory;
use Users\FrontModule\Components\OAuthUserConnect;
use Users\Model\Entities\User;
use Users\Model\Users;

class CustomerPresenter extends BasePresenter
{
	protected Orders                      $ordersService;
	protected Users                       $usersService;
	protected Customers                   $customersService;
	public Invoices                       $invoices;
	public IInvoicePdfBuilderFactory      $invoicePdfBuilderFactory;
	protected ProductsAssignedToCustomers $productsAssignedToCustomers;
	protected ProductsFacade              $productsFacade;

	public function __construct(
		Orders                      $orders,
		Users                       $users,
		Customers                   $customers,
		Invoices                    $invoices,
		IInvoicePdfBuilderFactory   $invoicePdfBuilder,
		ProductsAssignedToCustomers $productsAssignedToCustomers,
		ProductsFacade              $productsFacade
	)
	{
		$this->ordersService               = $orders;
		$this->usersService                = $users;
		$this->customersService            = $customers;
		$this->invoices                    = $invoices;
		$this->invoicePdfBuilderFactory    = $invoicePdfBuilder;
		$this->productsAssignedToCustomers = $productsAssignedToCustomers;
		$this->productsFacade              = $productsFacade;
	}

	public function checkRequirements($element): void
	{
		parent::checkRequirements($element);
		if (!$this->getUser()->isLoggedIn()) {
			$this->redirect(':Users:Front:Login:');
		}
	}

	protected function startup()
	{
		parent::startup();

		$this['meta']->setMeta('robots', 'noindex, nofollow');
	}

	public function handleOrderAttachment(int $orderId, string $filename): void
	{
		$order = $this->ordersService->getDetail($orderId);

		if (!$order || !$order->getCustomer()) {
			throw new BadRequestException();
		}

		/** @var User $user */
		$user              = $this->getUser()->getIdentity();
		$customer          = $order->getCustomer();
		$parentViewAllowed = false;

		if (
			EshopOrdersConfig::load('showOrdersOfChildrenUsers')
			&& (
				$user->childs->containsKey($order->getCustomer()->getUser()->getId())
				|| ($user->getParam('childUsers') && Arrays::contains($user->getParam('childUsers'), $order->getCustomer()->getUser()->getId()))
			)
		) {
			$parentViewAllowed = true;
		}

		if ($order->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent()
			|| ($order->getCustomer()->getId() !== $customer->getId() && !$parentViewAllowed)
		) {
			throw new BadRequestException();
		}

		$filePath = OrderHelper::getOrderAttachmentDir($orderId) . '/' . $filename;
		if (!file_exists($filePath)) {
			throw new BadRequestException;
		}

		$contentType = mime_content_type($filePath) ?: null;

		$this->sendResponse(new FileResponse($filePath, basename($filePath), $contentType, false));
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$this->template->messages      = $this->getTemplate()->flashes;
		$this->template->hasMyProducts = (bool) $this->productsAssignedToCustomers->getProductsIdByUser($this->getUser()->getId());
		$this->getTemplate()->flashes  = [];
	}

	public function actionOrderDetail(int $orderId): void
	{
		$order = $this->ordersService->getDetail($orderId);

		if (!$order) {
			throw new BadRequestException();
		}

		/** @var User $user */
		$user = $this->getUser()->getIdentity();

		$customer = $this->customersService->getByUser($user);

		$parentViewAllowed = false;
		if (
			EshopOrdersConfig::load('showOrdersOfChildrenUsers')
			&& (
				$user->childs->containsKey($order->getCustomer()->getUser()->getId())
				|| ($user->getParam('childUsers') && Arrays::contains($user->getParam('childUsers'), $order->getCustomer()->getUser()->getId()))
			)
		) {
			$parentViewAllowed = true;
		}

		if (
			!$order->getCustomer()
			|| $order->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent()
			|| ($order->getCustomer()->getId() !== $customer->getId() && !$parentViewAllowed)
		) {
			throw new BadRequestException();
		}

		$tmp = [];
		foreach ($order->getOrderItems() as $item) {
			if ($item->getProductId()) {
				$tmp[] = $item->getProductId();
			}
		}

		$tmp = array_unique($tmp);

		$products = !empty($tmp) ? $this->productsFacade->getProducts($tmp) : [];
		$this->productsFacade->productsService->loadDocuments($products);

		$this->template->showUser = EshopOrdersConfig::load('showOrdersOfChildrenUsers');
		$this->template->products = $products;
		$this->template->order    = $order;
	}

	public function actionMyProducts(): void
	{
		if (!Config::load('product.allowAssignProductToCustomer')) {
			$this->error();
		}

		$ids = $this->productsAssignedToCustomers->getProductsIdByUser($this->getUser()->getId());

		if (!empty($ids)) {
			$this->template->products = $this->productsFacade->getProducts($ids);
		} else {
			$this->template->products = [];
		}
	}

	public function actionEditProfile(): void
	{
	}

	public function actionInvoice(int $id): void
	{
		if (!EshopOrdersConfig::load('invoice.enable', false))
			throw new BadRequestException();

		/** @var Order $order */
		$order = $this->ordersService->getDetail($id);

		/** @var User $user */
		$user = $this->getUser()->getIdentity();

		$customer = $this->customersService->getByUser($user);

		$parentViewAllowed = false;
		if (
			EshopOrdersConfig::load('showOrdersOfChildrenUsers')
			&& (
				$user->childs->containsKey($order->getCustomer()->getUser()->getId())
				|| ($user->getParam('childUsers') && Arrays::contains($user->getParam('childUsers'), $order->getCustomer()->getUser()->getId()))
			)
		) {
			$parentViewAllowed = true;
		}

		if (!$order || !$order->getCustomer() || !$customer || ($order->getCustomer()->getId() != $customer->getId() && !$parentViewAllowed)) {
			throw new BadRequestException();
		}

		$invoice = $this->invoices->getOneByOrder($order, false);

		if (!$invoice) {
			throw new BadRequestException();
		}

		$this->invoicePdfBuilderFactory->create($invoice)->render();
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersList(IOrdersListFactory $factory): OrdersList
	{
		return $factory->create();
	}

	protected function createComponentProfileForm(IProfileFormFactory $factory): ProfileForm
	{
		/** @var User $user */
		$user    = $this->getUser()->getIdentity();
		$control = $factory->create($user);

		$control['form']->onSuccess['redirect'] = function() {
			$this->redirect('this');
		};

		return $control;
	}

	protected function createComponentAddressesList(IAddressesListFactory $factory): AddressesList
	{
		$control = $factory->create($this->getUser()->getId());

		$control->onAddressSave[] = function(AddressesList $control) {
			$this['profileForm']->redrawControl('formAddresses');
			$this['profileForm']->redrawControl('formAddresses2');
		};

		return $control;
	}

	protected function createComponentProductPreview(IProductPreviewFactory $factory): Multiplier
	{
		return new Multiplier(static function($id) use ($factory) {
			$control = $factory->create();
			$control->setProductById((int) $id);

			return $control;
		});
	}

	protected function createComponentOauthUserConnect(IOAuthUserConnectFactory $factory): OAuthUserConnect
	{
		return $factory->create();
	}
}
