<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities\Invoice;

use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\PriceCalculator\PriceCalculator;

/**
 * @ORM\Table("eshop_orders__invoice_data")
 * @ORM\Entity
 */
class InvoiceData
{
	use TId;

	/**
	 * @var ArrayCollection<Product>
	 * @ORM\OneToMany(targetEntity="Product", mappedBy="invoiceData", cascade={"persist"})
	 */
	public Collection $products;

	/**
	 * @ORM\OneToOne(targetEntity="Customer", cascade={"persist"})
	 * @ORM\JoinColumn(name="customer_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public ?Customer $customer = null;

	/**
	 * @var ArrayCollection<Discount>
	 *
	 * @ORM\OneToMany(targetEntity="Discount", mappedBy="invoiceData", cascade={"persist"})
	 */
	public Collection $discounts;

	/**
	 * @ORM\Column(type="string")
	 */
	public string $lang = 'cs';

	/**
	 * @ORM\Column(type="string")
	 */
	public ?string $currency = null;

	/**
	 * @ORM\Column(name="zero_vat", type="smallint", length=1, options={"default": 0})
	 */
	public int $zeroVat = 0;

	/**
	 * @ORM\OneToOne(targetEntity="Supplier", cascade={"persist"})
	 * @ORM\JoinColumn(name="supplier_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Supplier $supplier;

	/**
	 * @ORM\OneToOne(targetEntity="Spedition", inversedBy="invoiceData", cascade={"persist"})
	 * @ORM\JoinColumn(name="spedition_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Spedition $spedition;

	/**
	 * @ORM\OneToOne(targetEntity="Payment", inversedBy="invoiceData", cascade={"persist"})
	 * @ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Payment $payment;

	/**
	 * @ORM\OneToOne(targetEntity="EshopOrders\Model\Entities\Invoice", cascade={"persist"})
	 * @ORM\JoinColumn(name="invoice_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Invoice $invoice;

	public ?PriceCalculator $priceCalculator = null;

	public function __construct(
		Customer  $customer,
		Supplier  $supplier,
		Spedition $spedition,
		Payment   $payment,
		Invoice   $invoice
	)
	{
		$this->customer  = $customer;
		$this->supplier  = $supplier;
		$this->spedition = $spedition;
		$this->payment   = $payment;
		$this->invoice   = $invoice;
		$this->discounts = new ArrayCollection();
	}

	public function getCustomer(): Customer { return $this->customer; }

	public function getSupplier(): Supplier { return $this->supplier; }

	public function getSpedition(): Spedition { return $this->spedition; }

	public function getPayment(): Payment { return $this->payment; }

	public function getPrice(): float
	{
		if ($this->priceCalculator) {
			return $this->priceCalculator->getWithVatTotal()->getAmount()->toFloat();
		}

		$priceTotal = $this->getPriceItems();
		$priceTotal += $this->getSpedition()->getPrice();
		$priceTotal += $this->getPayment()->getPrice();

		foreach ($this->discounts as $discount) {
			if ($priceTotal < 0) {
				$priceTotal += abs((float) $discount->price);
			} else {
				$priceTotal -= abs((float) $discount->price);
			}
		}

		return (float) $priceTotal;
	}

	public function getPriceItems(): float
	{
		if ($this->priceCalculator) {
			return $this->priceCalculator->getWithVatItemsWithoutDiscount()->getAmount()->toFloat();
		}

		$priceTotal = 0;
		foreach ($this->products as $item) {
			$priceTotal += $item->getPrice() * $item->getQuantity();
		}

		return (float) $priceTotal;
	}

	public function getItemsPriceWithoutVat(): float
	{
		if ($this->priceCalculator) {
			return $this->priceCalculator->getWithoutVatItemsWithoutDiscount()->getAmount()->toFloat();
		}

		$priceTotal = 0;
		foreach ($this->products as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getPaySpedPrice(): float
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition()->getPrice();
		$priceTotal += $this->getPayment()->getPrice();

		return (float) $priceTotal;
	}

	public function getPaySpedPriceWithoutVat(bool $useCurrency = false): float
	{
		$priceTotal = 0;
		$priceTotal += $this->getSpedition()->getPriceWithoutVat();
		$priceTotal += $this->getPayment()->getPriceWithoutVat();

		return (float) $priceTotal;
	}

	public function getPaySpedId(): string { return $this->getSpedition()->speditionId . '-' . $this->getPayment()->paymentId; }

	public function getPaySpedName(): string { return $this->getSpedition()->name . ' - ' . $this->getPayment()->name; }

	/**
	 * @deprecated
	 */
	public function calculateDiscounts(float $price): float
	{
		foreach ($this->discounts as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	public function getPaySpedVatRate(): int { return $this->zeroVat ? 0 : $this->getSpedition()->getVatRate(); }

	/**
	 * @deprecated
	 */
	public function getBaseVatRates(bool $usePaySped = true): array
	{
		$list = [];

		$totalItemsPrice = $this->getPriceItems() + $this->getPaySpedPrice();
		$zeroPrice       = $totalItemsPrice === 0.0;

		foreach ($this->products as $item) {
			$vat = $item->getVatRate();

			if (!isset($list[$vat])) {
				$list[$vat] = [
					'withoutVat' => 0,
					'total'      => 0,
				];
			}

			if (!$zeroPrice) {
				$list[$vat]['withoutVat'] += $item->getTotalPriceWithoutVat();
				$list[$vat]['total']      += $item->getPriceTotal();
			}
		}

		if ($usePaySped) {
			if (!isset($list[$this->getPaySpedVatRate()])) {
				$list[$this->getPaySpedVatRate()] = [
					'withoutVat' => 0,
					'total'      => 0,
				];
			}

			$list[$this->getPaySpedVatRate()]['withoutVat'] += $this->getPaySpedPriceWithoutVat();
			$list[$this->getPaySpedVatRate()]['total']      += $this->getPaySpedPrice();
		}

		return $list;
	}

	public function getVatRates(): array
	{
		if ($this->priceCalculator) {
			return array_map(static function($vals) {
				return [
					'withoutVat' => $vals['withoutVat']->getAmount()->toFloat(),
					'total'      => $vals['total']->getAmount()->toFloat(),
					'rate'       => $vals['rate']->getAmount()->toFloat(),
				];
			}, $this->priceCalculator->getVatBreakdown());
		}

		// TODO remove
		$list = $this->getBaseVatRates();

		if ($this->getDiscountsByVat()) {
			foreach ($this->getDiscountsByVat() as $vat => $value) {
				$list[$vat]['withoutVat'] += $value['withoutVat'];
				$list[$vat]['total']      += $value['total'];
			}
		}

		foreach ($list as $vat => $prices) {
			$list[$vat]['rate'] = round($prices['total'] - $prices['withoutVat'], 2);
		}

		return $list;
	}

	/**
	 * @return Product[]
	 *
	 * @deprecated
	 */
	public function getProductsForDiscount(): array
	{
		$items = [];

		foreach ($this->products as $k => $product) {
			if ($product->getMoreDataValue('discountDisabled')) {
				continue;
			}

			$items[$k] = $product;
		}

		return $items;
	}

	public function getDiscountsByVat(): array
	{
		if ($this->priceCalculator) {
			return array_map(static function($vals) {
				return [
					'withoutVat' => $vals['withoutVat']->getAmount()->toFloat(),
					'total'      => $vals['total']->getAmount()->toFloat(),
				];
			}, $this->priceCalculator->getDiscountsByVat());
		}

		if (empty($this->discounts->toArray())) {
			return [];
		}

		$itemsPrice = 0;
		$dphRates   = [];

		foreach ($this->getProductsForDiscount() as $item) {
			$itemsPrice += $item->getPriceTotal();
		}

		$zeroPrice = $itemsPrice === 0;

		foreach ($this->getProductsForDiscount() as $item) {
			$vat = $item->getVatRate();
			if (!isset($dphRates[$vat])) {
				$dphRates[$vat] = 0;
			}

			if (!$zeroPrice) {
				$dphRates[$vat] += $item->getPriceTotal();
			}
		}

		$totalSale = 0;
		foreach ($this->discounts as $discount) {
			$totalSale += abs((float) $discount->price);
		}

		if ($this->invoice && $this->invoice->order && $this->invoice->order->isCorrectiveTaxDocument) {
			$totalSale = -$totalSale;
		}

		return OrderHelper::calculateVatWithSales($itemsPrice, $totalSale, $dphRates);
	}

	public function getDiscountsCodes(): array { return array_map(static fn(Discount $d) => $d->code, $this->discounts->toArray()); }
}
