<?php declare(strict_types = 1);

namespace EshopOrders\Model\Listeners;

use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreFlushEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\ORMException;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\OrderStatusHistory;
use EshopOrders\Model\Event\OrderStatusEvent;
use Nette\Security\IUserStorage;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Users\Model\Users;

class OrderStatusListener implements EventSubscriberInterface
{
	use SmartObject;

	protected EventDispatcher $eventDispatcher;
	protected Users           $usersService;
	protected IUserStorage    $userStorage;

	public function __construct(
		EventDispatcher $eventDispatcher,
		Users           $users,
		IUserStorage    $userStorage
	)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->usersService    = $users;
		$this->userStorage     = $userStorage;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param OrderStatus       $orderStatus
	 * @param PreFlushEventArgs $event
	 */
	public function postUpdateHandler(OrderStatus $orderStatus, PreFlushEventArgs $event): void
	{
		foreach ($event->getEntityManager()->getUnitOfWork()->getScheduledEntityInsertions() as $entity) {
			if ($entity instanceof OrderStatus) {
				$this->eventDispatcher->dispatch(new OrderStatusEvent($entity->getOrder(), $entity->getStatus()->getId()), Order::class . '::changeStatus');
			}
		}
	}

	/**
	 * @ORM\PostRemove
	 *
	 * @param LifecycleEventArgs $events
	 *
	 * @throws ORMException
	 */
	public function postRemoveHandler(OrderStatus $orderStatus, LifecycleEventArgs $events): void
	{
		$em      = $events->getEntityManager();
		$userRef = $this->usersService->getReference($this->userStorage->getIdentity()->getId());
		$e       = new OrderStatusHistory($orderStatus->getOrder(), $orderStatus->getStatus(), $userRef);
		$em->persist($e);
		$em->flush($e);
	}
}
