<?php declare(strict_types = 1);

namespace EshopOrders\Model\PriceCalculator;

use Brick\Math\RoundingMode;
use Brick\Money\Money;

class PriceCalculatorItem
{
	public ?int      $productId       = null;
	public ?int      $cartItemId      = null;
	public ?string   $name            = null;
	protected Money  $priceWithVat;
	protected ?Money $priceWithoutVat = null;
	protected int    $quantity;
	protected int    $vatRate;

	/** @var self[] */
	protected array $childs = [];

	protected bool $discountDisabled = false;

	public function __construct(
		string $name,
		float  $priceWithVat,
		int    $quantity,
		int    $vatRate,
		string $currency,
		bool   $discountDisabled = false
	)
	{
		$this->name             = $name;
		$this->priceWithVat     = Money::of($priceWithVat, $currency, null, RoundingMode::HALF_UP);
		$this->quantity         = $quantity;
		$this->vatRate          = $vatRate;
		$this->discountDisabled = $discountDisabled;
		$this->priceWithoutVat  = $this->priceWithVat->dividedBy(1 + $vatRate / 100, RoundingMode::HALF_UP);
	}

	public function addChild(string $k, self $child): void
	{
		$this->childs[$k] = $child;
	}

	public function isDiscountApplicable(): bool
	{
		return !$this->discountDisabled;
	}

	public function setQuantity(int $quantity): void
	{
		$this->quantity = $quantity;
	}

	public function getQuantity(): int
	{
		return $this->quantity;
	}

	public function getPriceWithVat(): Money
	{
		$price = $this->priceWithVat;

		foreach ($this->childs as $child) {
			$price = $price->plus($child->getTotalWithVat());
		}

		return $price;
	}

	public function getPriceWithoutVat(): Money
	{
		return $this->getTotalWithoutVat()->dividedBy($this->quantity, RoundingMode::HALF_UP);
	}

	public function getTotalWithoutVat(): Money
	{
		return $this->getPriceWithVat()
			->multipliedBy($this->quantity)
			->dividedBy(1 + ($this->vatRate / 100), RoundingMode::HALF_UP);
	}

	public function getVat(): Money
	{
		return $this->getTotalWithVat()->minus($this->getTotalWithoutVat());
	}

	public function getTotalWithVat(): Money
	{
		$price = $this->priceWithVat;

		foreach ($this->childs as $child) {
			$price = $price->plus($child->getTotalWithVat());
		}

		return $price->multipliedBy($this->quantity, RoundingMode::HALF_UP);
	}

	public function getVatRate(): float
	{
		return $this->vatRate;
	}
}
