<?php declare(strict_types = 1);

namespace EshopOrders\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\Event;
use Core\Model\Mailing\MailBuilderFactory;
use Core\Model\Mailing\TemplateFactory;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use EshopOrders\Model\Utils\Helpers;
use Mpdf\Output\Destination;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class InvoiceSubscriber implements EventSubscriberInterface
{
	protected Invoices                  $invoices;
	protected Translator                $translator;
	protected MailBuilderFactory        $mailer;
	protected Sellers                   $sellers;
	protected Sites                     $sites;
	protected TemplateFactory           $templateFactory;
	protected EventDispatcher           $eventDispatcher;
	protected IInvoicePdfBuilderFactory $invoicePdfBuilder;

	public function __construct(
		Invoices                  $invoices,
		Translator                $translator,
		MailBuilderFactory        $mailer,
		Sellers                   $sellers,
		Sites                     $sites,
		TemplateFactory           $templateFactory,
		EventDispatcher           $eventDispatcher,
		IInvoicePdfBuilderFactory $invoicePdfBuilder
	)
	{
		$this->invoices          = $invoices;
		$this->translator        = $translator;
		$this->mailer            = $mailer;
		$this->sellers           = $sellers;
		$this->sites             = $sites;
		$this->templateFactory   = $templateFactory;
		$this->eventDispatcher   = $eventDispatcher;
		$this->invoicePdfBuilder = $invoicePdfBuilder;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.invoices.checkOverdue' => 'checkOverdue',
		];
	}

	public function checkOverdue(Event $event): void
	{
		try {
			$now = new DateTime;

			$invoices = $this->invoices->getInvoicesOverdue();
			$this->eventDispatcher->dispatch(new Event(['data' => &$invoices]), self::class . '::onSelectOvedue');

			foreach ($invoices as $invoice) {
				$delay = abs((int) $now->diff((clone $invoice->dueDate)->modify('-1 day'))->format("%r%a"));;
				if ($delay % (int) EshopOrdersConfig::load('invoice.afterHowManyDaysSendTheOverdueNotice') !== 0) {
					continue;
				}

				$order = $invoice->order;

				$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/invoiceNotice.latte';
				$originFile = __DIR__ . '/../../FrontModule/_emails/invoiceNotice.latte';

				$this->translator->setLocale($order->lang);
				Sites::$currentIdentOverride = $order->lang;

				$mailer    = $this->mailer->create();
				$siteIdent = $order->site->getIdent();
				$mailer->setTemplate($this->templateFactory->create($siteIdent, $order->lang));
				$seller = $this->sellers->getSellerForSite($siteIdent);

				$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
				$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

				if (!file_exists($file)) {
					$file = $originFile;
				}

				$subject = $this->translator->translate('eshopOrders.emails.subjectInvoiceNotice',
					['orderId' => $order->getId(), 'siteName' => $mailer->getTemplate()->siteName]);

				$mailer->setTemplateFile($file);
				$mailer->setParameters([
					'subject'        => $subject,
					'order'          => $order,
					'orderId'        => $order->getId(),
					'originTemplate' => $originFile,
				]);

				$pdfBuilder = $this->invoicePdfBuilder->create($invoice);
				$mailer->message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));

				$mailer->setFrom($sellerEmail, $sellerName);
				$mailer->addTo($order->getAddressInvoice()->getEmail(), $order->getAddressInvoice()->getName());
				$mailer->send();
			}
		} catch (\Exception $e) {
			Debugger::log('Invoice notice email failed - ' . (isset($order) ? $order->getId() : ''), 'eshopOrderEmail');
			Debugger::log($e, 'eshopOrderEmail');
		}
	}

}
