<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Currencies;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class OrderDiscountForm extends BaseControl
{
	protected Order      $order;
	protected Currencies $currencies;

	public function __construct(
		Order      $order,
		Currencies $currencies
	)
	{
		$this->order      = $order;
		$this->currencies = $currencies;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$types = [
			OrderDiscount::TYPE_PERCENT => '%',
			OrderDiscount::TYPE_FIX     => 'Fix',
		];

		$defaultCurrency = $this->currencies->getDefaultCode();
		$currencies      = [
			$defaultCurrency => $defaultCurrency,
		];

		if ($this->order->currency && $this->order->currency->code !== $defaultCurrency) {
			$currencies[$this->order->currency->code] = $this->order->currency->code;
		}

		$form->addText('name', 'eshopOrders.orderDiscountForm.name')
			->setNullable();
		$form->addText('value', 'eshopOrders.orderDiscountForm.value')
			->setRequired()
			->addRule($form::FLOAT);
		$form->addSelect('saleType', null, $types)
			->setRequired();
		$form->addSelect('currency', 'eshopOrders.orderDiscountForm.currencyForFix', $currencies);

		$form->addSaveCancelControl();
		unset($form['saveControl']['save']);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$saleEntity = new OrderDiscount('', $this->order);
			$saleEntity->setName($values->name ?? '');
			$values->value   = (float) $values->value;
			$defaultCurrency = $this->currencies->getDefaultCode();

			if ($values->saleType === OrderDiscount::TYPE_PERCENT) {
				$totalPrice = $this->order->getPriceItems(false);
				$price      = $totalPrice * ($values->value / 100);
				$saleEntity->setPrice($price);
			} else if ($values->saleType === OrderDiscount::TYPE_FIX) {
				if ($values->currency !== $defaultCurrency && $this->order->currency) {
					$orderCurrency = $this->order->currency;
					$values->value = round(((float) $values->value) * $orderCurrency->rate, $orderCurrency->decimals);
				}

				$saleEntity->setPrice($values->value);
			}

			$saleEntity->setValue($values->value);
			$saleEntity->setType($values->saleType);

			$this->em->persist($saleEntity);
			$this->em->flush();

			$this->getPresenter()->flashMessageSuccess('default.saved');
		} catch (\Exception $exception) {
			Debugger::log($exception);
			$this->getPresenter()->flashMessageSuccess('default.error');
		}
	}

}
