<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\AdminModule\Components\Invoice\IQuickEditFormFactory;
use EshopOrders\AdminModule\Components\Invoice\QuickEditForm;
use EshopOrders\AdminModule\Components\OrderCurrency\IOrderCurrencyFormFactory;
use EshopOrders\AdminModule\Components\OrderCurrency\OrderCurrencyForm;
use EshopOrders\AdminModule\Components\OrderInvoice\IOrderInvoiceFormFactory;
use EshopOrders\AdminModule\Components\OrderInvoice\OrderInvoiceForm;
use EshopOrders\AdminModule\Model\OrderGifts;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Statuses;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;
use Users\Model\Users;

class OrderForm extends BaseControl
{
	public ?Order $order = null;

	protected Orders                                $ordersService;
	protected CardsPaymentService                   $cardPaymentService;
	protected Speditions                            $speditionsService;
	protected Payments                              $paymentsService;
	protected IOrderItemsGridFactory                $orderItemsGridFactory;
	protected IOrderStatusesGridFactory             $orderStatusesGridFactory;
	protected PriceFilter                           $priceFilter;
	protected OrderGifts                            $orderGifts;
	protected Sellers                               $sellersService;
	protected IIsPaidSwitchFactory                  $isPaidSwitchFactory;
	protected IEnableInvoiceGenerationSwitchFactory $enableInvoiceGenerationSwitchFactory;
	protected ICarrierHistoryGridFactory            $carrierHistoryGridFactory;
	protected Users                                 $usersService;
	protected Statuses                              $statusesService;
	protected IQuickEditFormFactory                 $editDUZPFormFactory;
	protected IOrderCurrencyFormFactory             $orderCurrencyFormFactory;
	protected IOrderInvoiceFormFactory              $orderInvoiceFormFactory;
	protected QrGenerator 						 $qrGenerator;

	public function __construct(
		Orders                                $orders,
		Speditions                            $speditions,
		Payments                              $payments,
		CardsPaymentService                   $cardsPaymentService,
		IOrderItemsGridFactory                $orderItemsGridFactory,
		IOrderStatusesGridFactory             $orderStatusesGridFactory,
		IIsPaidSwitchFactory                  $isPaidSwitchFactory,
		OrderGifts                            $orderGifts,
		Sellers                               $sellers,
		PriceFilter                           $priceFilter,
		IEnableInvoiceGenerationSwitchFactory $enableInvoiceGenerationSwitchFactory,
		ICarrierHistoryGridFactory            $carrierHistoryGridFactory,
		Users                                 $usersService,
		Statuses                              $statusesService,
		IQuickEditFormFactory                 $editDUZPFormFactory,
		IOrderCurrencyFormFactory             $orderCurrencyFormFactory,
		IOrderInvoiceFormFactory              $orderInvoiceFormFactory,
		QrGenerator 						  $qrGenerator
	)
	{
		$this->ordersService                        = $orders;
		$this->speditionsService                    = $speditions;
		$this->paymentsService                      = $payments;
		$this->orderItemsGridFactory                = $orderItemsGridFactory;
		$this->orderStatusesGridFactory             = $orderStatusesGridFactory;
		$this->cardPaymentService                   = $cardsPaymentService;
		$this->isPaidSwitchFactory                  = $isPaidSwitchFactory;
		$this->priceFilter                          = $priceFilter;
		$this->orderGifts                           = $orderGifts;
		$this->sellersService                       = $sellers;
		$this->carrierHistoryGridFactory            = $carrierHistoryGridFactory;
		$this->enableInvoiceGenerationSwitchFactory = $enableInvoiceGenerationSwitchFactory;
		$this->usersService                         = $usersService;
		$this->statusesService                      = $statusesService;
		$this->editDUZPFormFactory                  = $editDUZPFormFactory;
		$this->orderCurrencyFormFactory             = $orderCurrencyFormFactory;
		$this->orderInvoiceFormFactory              = $orderInvoiceFormFactory;
		$this->qrGenerator 						 = $qrGenerator;
	}

	public function render(): void
	{
		$this->template->paymentsService = $this->paymentsService;
		$this->template->order           = $this->order;
		$this->template->seller          = $this->sellersService->getSellerForSite($this->order->site->getIdent());
		$this->template->qr = $this->qrGenerator->getQrCodeByOrderAndSeller($this->order, $this->template->seller);
		$this->template->showPaid        = EshopOrdersConfig::load('ordersGrid.showCardHistory') && (
				EshopOrdersConfig::load('showPaidFieldForAllOrders', false) === true
				|| $this->order->getPaymentIdent() === 'card');

		if (EshopOrdersConfig::load('ordersGrid.showCardHistory') && $this->order->getPaymentIdent() == 'card') {
			$this->template->paymentHistory = $this->cardPaymentService->getHistoryOfOrder($this->order->getId());
		}

		$this->template->render($this->getTemplateFile());
	}

	public function setOrder($id)
	{
		$this->order = $this->ordersService->get($id);

		if (!$this->order) {
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
		}
	}

	public function redrawStatuses(): void
	{
		$this['orderStatusesGrid']['grid']->reload();
	}

	public function handleGiftReadyToDeliveryChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		if ($this->orderGifts->setReadyToDelivery((int) $id, (int) $newStatus))
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this->redrawControl('gifts');
			$this->redrawControl('expeditionForm');
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleMessageReadyToDeliveryChange($id, $newStatus): void
	{
		$presenter = $this->getPresenter();

		if ($this->ordersService->setMessageReadyToDelivery((int) $id, (bool) $newStatus))
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this->redrawControl('message');
			$this->redrawControl('expeditionForm');
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleRemoveGift($id): void
	{
		$presenter = $this->getPresenter();

		$orderGift = $this->orderGifts->get($id);
		$this->eventDispatcher->dispatch(new Event(['entity' => $orderGift]), 'eshopOrders.gifts.beforeDelete');

		$this->em->remove($orderGift);
		$this->em->flush();

		$presenter->flashMessageSuccess('default.removed');

		$this->redrawControl('gifts');
		$presenter->redrawControl('flashes');
	}

	public function handleValidate(string $type, string $val)
	{
		$presenter = $this->getPresenter();

		if ($type === 'idNumber') {
			$this->ordersService->validateIdNumber($this->order, (int) $val);
			$presenter->flashMessageSuccess('default.saved');
		} else if ($type === 'vatNumber') {
			$this->ordersService->validateVatNumber($this->order, (int) $val);
			$presenter->flashMessageSuccess('default.saved');
		} else {
			$presenter->flashMessageDanger('eshopOrders.order.validateNotFound');
		}

		$presenter->redirect('this');
	}

	public function handleChangePayment(int $id): void
	{
		$payment   = $this->paymentsService->get($id);
		$presenter = $this->getPresenter(false);

		if (!$presenter) {
			return;
		}

		if (!$payment) {
			$presenter->flashMessageDanger('eshopOrders.paymentNotFound');
		} else {
			try {
				$this->paymentsService->setOrderPayment(
					$this->order,
					$payment,
					$payment->getName()
				);
				$presenter->flashMessageSuccess('eshopOrders.orderPaymentForm.edited');
			} catch (\Exception $e) {
				$presenter->flashMessageDanger('default.error');
			}
		}

		$presenter->redrawControl('flashes');
		$this->redrawControl('payment');
		$this->redrawControl('spedition');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderItemsGrid()
	{
		$control = $this->orderItemsGridFactory->create($this->order->getId());

		return $control;
	}

	protected function createComponentOrderStatusesGrid()
	{
		$control = $this->orderStatusesGridFactory->create();
		if ($this->order) {
			$control->setOrder($this->order->getId());
		}

		return $control;
	}

	public function createComponentExpeditionForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)->setAjax();
		$form->addTextArea('message', 'eshopOrders.default.statusMessage');
		$form->addBool('sendEmailToCustomer', 'eshopOrders.default.sendEmailToCustomer')
			->setDefaultValue(1);

		$event                = new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null);
		$event->control       = $this;
		$event->data['order'] = $this->ordersService->get($this->order->getId());
		$this->eventDispatcher->dispatch($event, self::class . '::createExpeditionForm');

		$form->addSubmit('save', 'eshopOrders.itemsGrid.dispatch')
			->getControlPrototype()->addHtml('<i class="fa fa-paper-plane"></i>');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) {
			$this->em->beginTransaction();

			try {
				$values->status = OrderStatus::STATUS_SPEDITION;
				$order          = $this->ordersService->get($this->order->getId());
				$status         = $this->statusesService->getReference(OrderStatus::STATUS_SPEDITION);
				$userRef        = $this->usersService->getReference($this->getPresenter()->getUser()->id);

				$orderStatus = new OrderStatus($this->ordersService->getReference((int) $order->getId()), $status, $userRef);
				$orderStatus->setMessage($values->message);

				$event                   = new FormSuccessEvent(
					$form,
					$values,
					$this->getPresenter(false) ? $this->template : null,
					$this->getPresenter(false) ? $this->getPresenter() : null);
				$event->custom['entity'] = $orderStatus;
				$this->eventDispatcher->dispatch($event, OrderStatusForm::class . '::formSuccess');

				$this->em->persist($orderStatus)->flush($orderStatus);

				$event                  = new FormSuccessEvent($form, $values,
					$this->getPresenter(false) ? $this->template : null, $this->getPresenter(false) ? $this->getPresenter() : null,
				);
				$event->control         = $this;
				$event->custom['order'] = $order;
				$this->eventDispatcher->dispatch($event, self::class . '::expeditionFormSuccess');

				$this->em->commit();

				$this->getPresenter()->flashMessageSuccess('eshopOrders.statusForm.added');
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->rollback();
				}
				Debugger::log($e);

				return;
			}

			if ($values->sendEmailToCustomer) {
				try {
					if ($this->statusesService->sendOrderStatusEmail($order, $orderStatus)) {
						$this->getPresenter()->flashMessageSuccess('eshopOrders.statusForm.emailSent');
					}
				} catch (\Exception $e) {
					Debugger::log($e->getMessage(), 'orderStatusError');
					$this->getPresenter()->flashMessageDanger('eshopOrders.statusForm.emailError');

					return;
				}
			}

			$this->getPresenter()->redirect(':EshopOrders:Admin:Default:');
		};

		return $form;
	}

	protected function createComponentCarrierHistoryGrid(): CarrierHistoryGrid
	{
		return $this->carrierHistoryGridFactory->create($this->order);
	}

	protected function createComponentInvoiceQuickEditForm(): QuickEditForm
	{
		return $this->editDUZPFormFactory->create($this->order->getInvoice());
	}

	protected function createComponentCurrencyChangeForm(): OrderCurrencyForm
	{
		return $this->orderCurrencyFormFactory->create($this->order);
	}

	protected function createComponentOrderInvoiceForm(): OrderInvoiceForm
	{
		return $this->orderInvoiceFormFactory->create($this->order);
	}

	protected function createComponentIsPaidSwitch(): IsPaidSwitch { return $this->isPaidSwitchFactory->create($this->order); }

	protected function createComponentEnableInvoiceGenerationSwitch(): EnableInvoiceGenerationSwitch { return $this->enableInvoiceGenerationSwitchFactory->create($this->order); }

	protected function createComponentInternalNoteForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);

		$form->addTextArea('note')
			->setDefaultValue($this->order->getParam(Order::paramInternalNote))
			->setHtmlAttribute('rows', $this->order->getMessage() ? 3 : 7);
		$form->addSubmit('save', 'default.save', 'btn-success btn-xs');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): bool {
			try {
				if (!EshopOrdersConfig::load('orderForm.allowInternalNote')) {
					throw new \Exception('Not allowed');
				}

				if ($values->note) {
					$this->order->setParam(Order::paramInternalNote, $values->note);
				} else {
					$this->order->removeParam(Order::paramInternalNote);
				}

				$this->em->persist($this->order);
				$this->em->flush();

				$this->presenter->flashMessageSuccess('default.saved');
			} catch (\Exception $e) {
				$this->presenter->flashMessageDanger('default.error');
			}

			$this->presenter->redrawControl('flashes');

			return false;
		};

		return $form;
	}
}
