<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\Application\AppState;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Event\Event;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings as CoreStrings;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Currency\DI\CurrencyExtension;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\AvailabilityService;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Availability;
use EshopOrders\AdminModule\Components\Expedition\IExpeditionAdvancedOptionsFormFactory;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Orders;
use EshopOrders\Model\OrderStatuses;
use EshopOrders\Model\Utils\Strings;
use Nette\Application\IPresenter;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Ublaboo\DataGrid\Column\Action\Confirmation\StringConfirmation;

class OrdersGrid extends BaseControl
{
	/** @var string @persistent */
	public $status = 'all';

	protected array $ids             = [];
	protected array $exportCallbacks = [];

	/** @var OrderAddress[] */
	protected array $cAddresses = [];

	/** @var OrderDiscount[][] */
	protected array $cOrdersDiscounts = [];

	/** @var OrderSpedition[] */
	protected array $cOrdersSpeditions = [];

	/** @var OrderPayment[] */
	protected array $cOrdersPayments = [];

	protected Orders                                $ordersService;
	protected Speditions                            $speditionsService;
	protected Payments                              $paymentsService;
	protected Statuses                              $statusesService;
	protected OrderStatuses                         $orderStatusesService;
	protected Sites                                 $sitesService;
	protected SessionSection                        $sessionSection;
	protected PriceFilter                           $priceFilter;
	protected Sellers                               $sellers;
	protected GroupsCustomers                       $groupsCustomers;
	protected IExpeditionAdvancedOptionsFormFactory $advancedOptionsFormFactory;
	protected IOrderMessageModalFactory             $orderMessageModalFactory;
	protected AvailabilityService                   $availabilityService;

	public function __construct(
		Orders                                $orders,
		Speditions                            $speditions,
		Payments                              $payments,
		Statuses                              $statuses,
		Sites                                 $sites,
		PriceFilter                           $priceFilter,
		Session                               $session,
		Sellers                               $sellers,
		OrderStatuses                         $orderStatusesService,
		GroupsCustomers                       $groupsCustomers,
		IExpeditionAdvancedOptionsFormFactory $advancedOptionsFormFactory,
		IOrderMessageModalFactory             $orderMessageModalFactory,
		AvailabilityService                   $availabilityService
	)
	{
		$this->ordersService              = $orders;
		$this->speditionsService          = $speditions;
		$this->paymentsService            = $payments;
		$this->statusesService            = $statuses;
		$this->sitesService               = $sites;
		$this->priceFilter                = $priceFilter;
		$this->sessionSection             = $session->getSection('eshopOrdersOrdersGrid');
		$this->status                     = EshopOrdersConfig::load('ordersGrid.defaultStatus');
		$this->sellers                    = $sellers;
		$this->orderStatusesService       = $orderStatusesService;
		$this->groupsCustomers            = $groupsCustomers;
		$this->advancedOptionsFormFactory = $advancedOptionsFormFactory;
		$this->orderMessageModalFactory   = $orderMessageModalFactory;
		$this->availabilityService        = $availabilityService;

		if (EshopOrdersConfig::load('ordersGrid.useTabs')) {
			$this->monitor(IPresenter::class, function() {
				$this->status = $this->sessionSection->status ?: 'created';
			});
		}
	}

	public function render(): void
	{
		if (EshopOrdersConfig::load('ordersGrid.useTabs')) {
			$statuses                     = $this->statusesService->getForSelectOption();
			$this->template->statuses     = $statuses;
			$this->template->activeStatus = $this->status;
		}

		$this->template->render($this->getTemplateFile());
	}

	public function handleChangeStatus(string $s): void
	{
		$this->status                 = $s;
		$this->sessionSection->status = $s;
		$this->redrawControl('nav');
		$this->redrawControl('gridWrap');
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false)
			->setItemsPerPageList([20, 50, 100, 200], EshopOrdersConfig::load('ordersGrid.allowShowAllItems', false))
			->setDefaultPerPage(50);
		$grid->setAutoSubmit((bool) EshopOrdersConfig::load('ordersGrid.gridAutoSubmit', true));
		$grid->detailEyePosition = (string) EshopOrdersConfig::load('ordersGrid.detailEyePosition', 'right');

		$qb = $this->ordersService->getEr()->createQueryBuilder('o')
			->orderBy('o.id', 'DESC')
			->groupBy('o.id');

		$join = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('os2')
			->select('GROUP_CONCAT(IDENTITY(os2.status))')
			->andWhere('os2.order = o.id')
			->andWhere('os2.deleted IS NULL OR os2.deleted = 0')
			->groupBy('os2.order')
			->orderBy('os2.order', 'DESC')
			->getQuery();

		if (EshopOrdersConfig::load('ordersGrid.useTabs')) {
			$status = $this->getParameter('s') ?: $this->status;
			if (array_key_exists($status, $this->statusesService->getForSelectOption())) {
				$qb->addSelect("(" . $join->getDQL() . ") as hidden statusFilter")
					->andHaving('statusFilter LIKE :status2')
					->setParameter('status2', "%" . $status);
			}
		}

		/** @var OrderStatus[][] $orderStatuses */
		$orderStatuses = [];

		/** @var OrderItem[][] $orderItems */
		$orderItems = [];

		/** @var Invoice $invoices */
		$invoices = [];

		/** @var Order[] $ordersBySpedition */
		$ordersBySpedition = [];

		$orderDeliveryCountries = [];

		$customersGroupIds = [];
		$allPayments       = $this->paymentsService->getAll();

		$grid->setDataSource(new DoctrineDataSource($qb, 'o.id'));

		$grid->getDataSource()->onDataLoaded[] = function($items)
		use (&$orderStatuses, &$orderItems, &$invoices, $grid, &$ordersBySpedition, &$customersGroupIds, &$orderDeliveryCountries) {
			$ids                  = [];
			$invoiceIds           = [];
			$allSpedition         = $this->speditionsService->getAll();
			$addressesIds         = [];
			$deliveryAddressesIds = [];
			$customerIds          = [];

			$ids       = array_map(static fn(Order $order) => $order->getId(), $items);
			$this->ids = $ids;

			$this->em->getRepository(OrderSpedition::class)->findBy(['order' => $ids]);
			$this->em->getRepository(OrderPayment::class)->findBy(['order' => $ids]);

			/** @var Order[] $items */
			foreach ($items as $k => $v) {
				$spedition = $allSpedition[$v->getEshopSpeditionId()] ?? null;
				if ($spedition) {
					$ordersBySpedition[$spedition->getIdent()][$v->getId()] = $v;
				}

				if ($v->getCustomer()) {
					$customerIds[] = $v->getCustomer()->getId();
				}

				if ($v->getInvoice()) {
					$invoiceIds[$v->getInvoice()->getId()] = $v->getId();
				}

				if ($v->getAddressInvoiceRaw()) {
					$addrId = $v->getAddressInvoiceRaw()->getId();

					$addressesIds[]                    = $addrId;
					$deliveryAddressesIds[$v->getId()] = $addrId;
				}

				if ($v->getAddressDeliveryRaw()) {
					$addrId = $v->getAddressDeliveryRaw()->getId();

					$addressesIds[]                    = $addrId;
					$deliveryAddressesIds[$v->getId()] = $addrId;
				}
			}

			// Doručovací adresy
			if (!empty($deliveryAddressesIds)) {
				$tmp = array_flip($deliveryAddressesIds);
				foreach ($this->em->getRepository(OrderAddress::class)->createQueryBuilder('oa')
					         ->select('oa.id as id, c.id as country')
					         ->innerJoin('oa.country', 'c')
					         ->where('oa.id IN (' . implode(',', $deliveryAddressesIds) . ')')
					         ->getQuery()
					         ->enableResultCache(60)
					         ->getResult() as $row) {
					$orderDeliveryCountries[$tmp[$row['id']]] = $row['country'];
				}
			}

			// Adresy
			if (!empty($addressesIds)) {
				$addressesIds = array_unique($addressesIds);
				foreach ($this->em->getRepository(OrderAddress::class)->createQueryBuilder('oa')
					         ->addSelect('c')
					         ->leftJoin('oa.country', 'c')
					         ->where('oa.id IN (' . implode(',', $addressesIds) . ')')
					         ->getQuery()
					         ->enableResultCache(60)
					         ->getResult() as $row) {
					/** @var OrderAddress $row */
					$this->cAddresses[$row->getId()] = $row;
				}
			}

			// Skupiny zakazniku
			if (!empty($customerIds) && EshopOrdersConfig::load('ordersGrid.showCustomerGroup', true)) {
				foreach ($this->em->getRepository(Customer::class)->createQueryBuilder('c')
					         ->select('c.id, IDENTITY(c.groupCustomers) as groupId')
					         ->where('c.id IN (' . implode(',', array_unique($customerIds)) . ')')
					         ->getQuery()
					         ->enableResultCache(60)
					         ->getArrayResult() as $row) {
					$customersGroupIds[$row['id']] = $row['groupId'];
				}
			}

			if (!empty($ids)) {
				// Slevy objednavky
				foreach ($this->em->getRepository(OrderDiscount::class)->createQueryBuilder('od')
					         ->where('od.order IN (' . implode(',', $ids) . ')')
					         ->getQuery()
					         ->enableResultCache(60)
					         ->getResult() as $row) {
					$this->cOrdersDiscounts[$row->getOrder()->getId()][$row->getId()] = $row;
				}
			}

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->addSelect('s')
				         ->innerJoin('os.status', 's')
				         ->where('os.order IN (:orders)')->setParameter('orders', $ids)
				         ->orderBy('os.created', 'DESC')
				         ->getQuery()
				         ->enableResultCache(15)
				         ->getResult() as $row) {
				/** @var OrderStatus $row */
				if ($row->isDeleted())
					continue;

				$orderStatuses[$row->getOrder()->getId()][] = $row;
			}

			$prodIds = [];
			foreach ($this->em->getRepository(OrderItem::class)->createQueryBuilder('oi')
				         ->addSelect('s')
				         ->where('oi.order IN (:orders)')->setParameter('orders', $ids)
				         ->leftJoin('oi.sales', 's')
				         ->getQuery()->getResult() as $row) {
				/** @var OrderItem $row */
				$orderItems[$row->getOrder()->getId()][] = $row;
				$prodIds[]                               = $row->getProductId();
			}

			$grid->template->orderItems = $orderItems;

			if (EshopOrdersConfig::load('invoice.enable')) {
				foreach ($this->em->getRepository(Invoice::class)->createQueryBuilder('i')
					         ->where('i.id IN (:ids)')
					         ->setParameter('ids', array_keys($invoiceIds))
					         ->getQuery()
					         ->setHint(Query::HINT_FORCE_PARTIAL_LOAD, true)
					         ->getResult() as $row) {
					/** @var Invoice $row */
					$invoices[$invoiceIds[$row->getId()]] = $row;
				}
			}

			$grid->template->loadItemsMoreData = function($items) {
				$event                     = new Event();
				$event->data['orderItems'] = &$items;
				$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');
			};
		};

		$customerLinkFn = fn($col) => function(Order $order) use ($col): Html {
			$addr = $this->getAddress($order->getAddressInvoiceRaw());
			$html = Html::el();

			if ($addr) {
				$html->setText($col === 'firstName' ? $addr->getFirstName() : $addr->getLastName());

				if ($order->getCustomer()) {
					$html->setName('a')
						->setAttribute('href', $this->presenter->link('Customers:editCustomer', $order->getCustomer()->getId()));
				}
			}

			return $html;
		};

		//Columns
		$grid->addColumnText('id', 'eshopOrders.default.orderNumberShort')->setAlign('right')
			->setRenderer(function(Order $order) use (&$ordersBySpedition, &$orderDeliveryCountries) {
				$html = Html::el('div')
					->addHtml(Html::el('a', ['href' => $this->getPresenter()->link('Default:editOrder', $order->getId())])
						->setText($order->getId()));

				$icons = Html::el('div', [
					'class' => 'd-flex justify-content-end align-items-baseline mt-1',
					'style' => 'gap: 4px; line-height: 1',
				]);

				$addr = $order->getAddressInvoiceRaw() ? $this->cAddresses[$order->getAddressInvoiceRaw()->getId()] : null;
				if ($addr && $addr->getCountry() && EshopOrdersConfig::load('highlightOtherCountryVat') && OrderHelper::isOtherCountryVatFilled($addr->getCountry()->getId(), $addr->getVatNumber())) {
					$icons->addHtml('<i class="fab fa-creative-commons-zero text-danger" title="' . $this->t('eshopOrders.otherCountryVatFilled') . '"></i>');
				}

				$country = $orderDeliveryCountries[$order->getId()] ?? null;
				if ($country && Arrays::contains(EshopOrdersConfig::load('highlightDeliveryCountryInOrdersGrid'), Strings::upper($country))) {
					$icons->addHtml('<i class="text-success">' . Strings::lower($country) . '</i>');
				}

				if ($icons->getChildren()) {
					$html->addHtml($icons);
				}

				$event = new Event([
					'html'              => $html,
					'order'             => $order,
					'allIds'            => $this->ids,
					'ordersBySpedition' => $ordersBySpedition,
				]);
				$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::columnIdRender');

				return $html;
			});

		if (EshopOrdersConfig::load('ordersGrid.showCompany')) {
			$grid->addColumnText('company', 'eshopOrders.default.company')->setRenderer(function(Order $row) {
				$addr = $this->getAddress($row->getAddressInvoiceRaw());

				return $addr ? $addr->getCompany() : '';
			})->setFilterText()
				->setCondition(function(QueryBuilder $qb, $value): void {
					$this->joinFilterAddressInvoice($qb);

					$qb->andWhere('ai.company LIKE :company OR ad.company LIKE :company')
						->setParameter('company', "%$value%");
				});
		}

		$grid->addColumnText('firstName', 'default.name', 'addressInvoice.firstName')
			->setRenderer($customerLinkFn('firstName'))
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $value): void {
				$this->joinFilterAddressInvoice($qb);
				$this->joinFilterAddressDelivery($qb);

				$qb->andWhere('ai.firstName LIKE :firstName OR ad.firstName LIKE :firstName')
					->setParameter('firstName', "%$value%");
			});

		$grid->addColumnText('lastName', 'eshopOrders.default.lastName', 'addressInvoice.lastName')
			->setRenderer($customerLinkFn('lastName'))
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $value): void {
				$this->joinFilterAddressInvoice($qb);
				$this->joinFilterAddressDelivery($qb);

				$qb->andWhere('ai.lastName LIKE :lastName OR ad.lastName LIKE :lastName')
					->setParameter('lastName', "%$value%");
			});

		$grid->addColumnText('email', 'eshopOrders.default.email')->setRenderer(function(Order $row): Html {
			$addr = $this->getAddress($row->getAddressInvoiceRaw());
			$html = Html::el();

			if ($addr) {
				$html->setName('a')
					->setAttribute('href', 'mailto:' . $addr->getEmail())
					->setAttribute('target', '_blank')
					->setAttribute('title', (string) $addr->getEmail())
					->setText(Strings::truncate((string) $addr->getEmail(), 25));
			}

			return $html;
		})
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $value): void {
				$this->joinFilterAddressInvoice($qb);
				$this->joinFilterAddressDelivery($qb);

				$qb->andWhere('ai.email LIKE :email OR ad.email LIKE :email')
					->setParameter('email', "%$value%");
			});

		if (EshopOrdersConfig::load('ordersGrid.showIco', false)) {
			$grid->addColumnText('idNumber', 'eshopOrders.default.idNumber')
				->setFitContent()
				->setRenderer(function(Order $row): string {
					$addr = $row->getAddressDelivery();

					return $addr && $addr->getIdNumber() ? $addr->getIdNumber() : '';
				})
				->setFilterText()
				->setCondition(function(QueryBuilder $qb, $value): void {
					$this->joinFilterAddressInvoice($qb);
					$qb->andWhere('ai.idNumber LIKE :idNumber')
						->setParameter('idNumber', "%$value%");
				});
		}

		$grid->addColumnText('phone', 'eshopOrders.default.phone')
			->setRenderer(function(Order $row): string {
				$addr = $this->getAddress($row->getAddressInvoiceRaw());

				return $addr && $addr->getPhone()
					? CoreStrings::phoneFormat($addr->getPhone(), $addr->getCountry() ? $addr->getCountry()->getId() : null, false)
					: '';
			})
			->setFilterText()
			->setCondition(function(QueryBuilder $qb, $value): void {
				$this->joinFilterAddressInvoice($qb);
				$this->joinFilterAddressDelivery($qb);

				$qb->andWhere('ai.phone LIKE :phone OR ad.phone LIKE :phone')
					->setParameter('phone', "%$value%");
			});

		if (EshopOrdersConfig::load('ordersGrid.showCustomerGroup', true)) {
			$grid->addColumnText('cGroup', str_replace(' ', '<br />', $this->t('eshopOrders.default.customerGroup')))
				->setHeaderEscaping(true)
				->setRenderer(function(Order $row) use (&$customersGroupIds): string {
					$customerId = $row->getCustomer() ? $row->getCustomer()->getId() : null;

					if ($customerId) {
						$groupId = $customersGroupIds[$customerId] ?? null;

						if ($groupId) {
							return $this->groupsCustomers->getOptionsForSelectShort()[$groupId] ?: '';
						}
					}

					return '';
				})->setAlign('center')
				->setFilterSelect(['' => ''] + ['withoutGroup' => $this->t('eshopOrders.default.withoutGroup')] + $this->groupsCustomers->getOptionsForSelect())
				->setCondition(function(QueryBuilder $qb, $value): void {
					$qb->leftJoin('o.customer', 'customer');
					$qb->leftJoin('customer.groupCustomers', 'cGroup');

					if ($value === 'withoutGroup') {
						$qb->andWhere('cGroup IS NULL');
					} else {
						$qb->andWhere('cGroup = :group')
							->setParameter('group', $value);
					}
				});


			$grid->getColumn('cGroup')->getElementPrototype('th')->addClass('w1nw');
			$grid->getColumn('cGroup')->getElementPrototype('td')->addClass('w1nw');
		}

		if (EshopOrdersConfig::load('ordersGrid.showMessage')) {
			$grid->addColumnText('message', 'eshopOrders.default.message')
				->setRenderer(function(Order $order) {
					if (!$order->getMessage())
						return '';

					return Html::el('a', ['href'  => $this->link('orderMessageModal!', $order->getId()),
					                      'class' => 'btn btn-success btn-xs ajax'])
						->addHtml(Html::el('i', ['class' => 'fas fa-comment']));
				})->setFitContent()
				->setAlign('center');
		}

		$grid->addColumnText('spedition', 'eshopOrders.default.spedition', 'spedition.name')
			->setRenderer(function(Order $order) use (&$ordersBySpedition): Html {
				if (!$order->getSpedition()) {
					return Html::el('');
				}

				$html = Html::el()->setText($order->getSpedition()->getName());

				$event = new Event([
					'html'              => $html,
					'order'             => $order,
					'allIds'            => $this->ids,
					'ordersBySpedition' => $ordersBySpedition,
				]);
				$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::columnSpeditionRender');

				return $html;
			});

		$grid->addColumnText('payment', 'eshopOrders.default.payment', 'payment.name')
			->setFilterSelect(['' => ''] + $this->paymentsService->getForSelectOption(), 'payment.payment')
			->setCondition(function(QueryBuilder $qb, $value) {
				if (empty($value)) {
					return;
				}

				$qb->innerJoin('o.payment', 'payment');

				$criteria = Criteria::create();
				$criteria->orWhere(Criteria::expr()->eq('payment.payment', $value));
				$this->eventDispatcher->dispatch(new GridFilterEvent($qb, $criteria, $value), OrdersGrid::class . '::filterPayment');
				$qb->addCriteria($criteria);
			});

		if (EshopOrdersConfig::load('ordersGrid.showSpeditionDate')) {
			$grid->addColumnText('speditionTime', 'eshopOrders.defaultGrid.speditionTime')
				->setRenderer(function(Order $row) use (&$orderStatuses) {
					foreach (array_reverse($orderStatuses[$row->getId()]) as $status) {
						/** @var OrderStatus $status */
						if ($status->getStatus()->getId() === OrderStatus::STATUS_SPEDITION) {
							return $status->getCreated()->format('j. n. Y H:i');
						}
					}

					return '';
				})->getElementPrototype('th')->addClass('w1nw');


			$grid->addFilterDateRange('speditionTime', '')
				->setFormat('j. n. Y', 'j. n. Y')
				->setCondition(function(QueryBuilder $qb, $values) {
					if ($values->from || $values->to) {
						$qb->innerJoin('o.orderStatuses', 'oSpeditionTime', Join::WITH, 'oSpeditionTime.status = :speditionStatusName')
							->setParameter('speditionStatusName', OrderStatus::STATUS_SPEDITION);
					}

					if ($values->from) {
						$date = DateTime::createFromFormat('j. n. Y', $values->from);
						if ($date instanceof \DateTimeInterface) {
							$qb->andWhere('oSpeditionTime.created >= :oSpeditionTimeFrom')
								->setParameter('oSpeditionTimeFrom', $date->setTime(0, 0, 0));
						}
					}

					if ($values->to) {
						$date = DateTime::createFromFormat('j. n. Y', $values->to);
						if ($date instanceof \DateTimeInterface) {
							$qb->andWhere('oSpeditionTime.created <= :oSpeditionTimeTo')
								->setParameter('oSpeditionTimeTo', $date->setTime(23, 59, 59));
						}
					}
				});
		}

		$grid->addColumnDateTime('created', 'eshopOrders.defaultGrid.createdTime', 'createdTime')
			->setRenderer(function(Order $row) use (&$orderStatuses, &$invoices) {
				if ($row->isCorrectiveTaxDocument && isset($invoices[$row->getId()])) {
					return $invoices[$row->getId()]->createdAt->format('d. m. Y H:i');
				}

				$statuses = $orderStatuses[$row->getId()];

				return $statuses ? end($statuses)->getCreated()->format('d. m. Y H:i') : '';
			})->getElementPrototype('th')->addClass('w1nw');

		// Stavy objednavky
		$statusesShowSeparated  = EshopOrdersConfig::load('ordersGrid.showStatusesSeparated', false);
		$statusesUseShortTitles = EshopOrdersConfig::load('ordersGrid.showShortStatusNames', false);
		$statusesTitlePrefix    = 'eshopOrders.ordersGrid.' . ($statusesUseShortTitles ? 'statusesShort.' : 'statusesLong.');
		if (is_array($statusesShowSeparated)) {
			foreach ($statusesShowSeparated as $key) {
				$grid->addColumnText('status' . ucfirst($key), $statusesTitlePrefix . $key)
					->setRenderer(function(Order $row) use ($key, &$orderStatuses, &$allPayments) {
						$status   = false;
						$statuses = $orderStatuses[$row->getId()];

						if (!$statuses[0]) {
							return '';
						}

						$payment      = $allPayments[$row->getEshopPaymentId() ?: 0] ?? null;
						$paymentIdent = $payment ? $payment->getIdent() : null;

						$lastStatus = $statuses[0]->getStatus()->getId();

						if ($lastStatus === OrderStatus::STATUS_CANCELED) {
							$status = 'bg-success';
						} else {
							if ($key === OrderStatus::STATUS_IS_PAID) {
								if ($row->isPaid) {
									$status = 'bg-success';
								} else if (in_array((string) $paymentIdent, ['card', 'transfer', 'pickup',
									'store', 'storeCash', 'storeCard'])) {
									$status = 'bg-danger';
								}
							} else if ($key === OrderStatus::STATUS_SPEDITION && $lastStatus === OrderStatus::STATUS_PROCESSING) {
								$status = 'bg-warning';
							} else {
								foreach ($statuses as $orderStatus) {
									if ($orderStatus->getStatus()->getId() === $key) {
										$status = 'bg-success';
										break;
									}
								}
							}
						}

						if ($status) {
							return Html::el('div', [
								'class' => 'status-circle ' . $status,
							]);
						}

						return '';
					})
					->setAlign('center')
					->setFilterSelect([
						''  => '',
						'0' => $this->t('default.no'),
						'1' => $this->t('default.yes'),
					])->setCondition(function(QueryBuilder $qb, $value) use ($key) {
						if ($key === 'isPaid') {
							$qb->andWhere('o.isPaid = :osIsPaid')
								->setParameter('osIsPaid', $value);
						} else {
							$queryKey = 'os' . ucfirst($key);

							if ($value) {
								$qb->innerJoin('o.orderStatuses', $queryKey, Join::WITH, "$queryKey.status = :$queryKey")
									->setParameter($queryKey, $key);
							} else {
								if (!in_array('osConcat', $qb->getAllAliases()))
									$qb->leftJoin('o.orderStatuses', 'osConcat')->addSelect('GROUP_CONCAT(IDENTITY(osConcat.status)) as hidden statusesString');
								$qb->andHaving("statusesString NOT LIKE :$queryKey")
									->setParameter($queryKey, "%$key%");
							}
						}
					});
			}
		}

		if (EshopOrdersConfig::load('ordersGrid.useTabs', false) === false
			&& EshopOrdersConfig::load('ordersGrid.showCurrentStatus', true) === true) {
			$grid->addColumnText('status', 'eshopOrders.default.status')->setRenderer(function(Order $row) use (&$orderStatuses) {
				$statuses = $orderStatuses[$row->getId()];

				return $statuses ? $statuses[0]->getStatus()->getName() : '';
			});
		}

		if (EshopOrdersConfig::load('ordersGrid.showCardHistory')
			&& (!is_array($statusesShowSeparated) || !in_array('isPaid', $statusesShowSeparated))) {
			$showPaidFieldForAll = EshopOrdersConfig::load('showPaidFieldForAllOrders', false);
			$grid->addColumnText('isPaid', 'eshopOrders.default.' . ($showPaidFieldForAll ? 'isPaid' : 'isPaidWithCard'))
				->setRenderer(function(Order $row) use ($showPaidFieldForAll, &$allPayments) {
					$payment = $allPayments[$row->getEshopPaymentId() ?: 0] ?? null;

					if (!($showPaidFieldForAll || ($payment && $payment->getIdent() === 'card'))) {
						return '';
					}

					$isPaid = $row->isPaid;

					return Html::el('div', [
						'class' => 'btn btn-xs ' . ($isPaid ? 'btn-success' : 'btn-danger'),
					])->setText($this->t('default.' . ($isPaid ? 'yes' : 'no')));
				})
				->setAlign('center')
				->setFilterSelect([
					''  => '',
					'0' => $this->t('default.no'),
					'1' => $this->t('default.yes'),
				]);
		}

		$grid->addColumnNumber('price', 'eshopOrders.default.price')->setRenderer(function(Order $row) use (&$orderItems) {
			if ($orderItems[$row->getId()]) {
				$row->setOrderItems($orderItems[$row->getId()]);
			}

			$row->orderDiscounts = new ArrayCollection($this->cOrdersDiscounts[$row->getId()] ?? []);

			if ($row->currency) {
				$price2    = $row->getPrice(true);
				$code2     = $row->currency->code;
				$decimals2 = $row->currency->decimals;
			}

			return $this->priceFilter
				->renderTwoPrices($row->getPrice(), $row->getDefaultCurrency(),
					$price2 ?? null, $code2 ?? null, $decimals2 ?? null,
					$row->getPriceWithoutVat(), $this->t('eshopOrders.orderForm.noVat'),
					$row->getPriceWithoutVat(true));
		})->getElementPrototype('th')->addClass('w1nw');

		if (class_exists(CurrencyExtension::class)) {
			$grid->addColumnText('currency', 'eshopOrders.default.currency', 'currencyCode');
		}

		$sites = $this->sitesService->getIdents(false);
		if (count($sites) > 1) {
			$grid->addColumnText('site', 'eshopOrders.default.eshop', 'site.ident')
				->getElementPrototype('td')->addClass('w1nw');

			$tmp     = [];
			$sellers = [];
			foreach ($this->sellers->getSitesInUse() as $row) {
				$sellers[$row['sellerName']][] = $row['site'];
			}

			foreach (EshopOrdersConfig::load('ordersGrid.extraSellersFilter', []) as $k => $v) {
				$sellers[$k] = $v;
			}

			foreach ($sellers as $k => $v) {
				$tmp['sellers_' . Json::encode($v)] = $k;
			}

			$grid->getColumn('site')->setFilterSelect([
				null                                       => '',
				$this->t('eshopOrders.ordersGrid.eshops')  => $sites,
				$this->t('eshopOrders.ordersGrid.sellers') => $tmp,
			])->setCondition(function(QueryBuilder $qb, $value) {
				if (Strings::startsWith($value, 'sellers_')) {
					$json = substr($value, 8);
					if (!Arrays::isJson($json)) {
						return;
					}

					$or = [];
					foreach (Json::decode($json, Json::FORCE_ARRAY) as $v) {
						if (is_string($v)) {
							$or[] = "o.site = '{$v}'";
						} else {
							$vAnd = [];
							if (isset($v['site'])) {
								$vAnd[] = "o.site = '{$v['site']}'";
							}

							if (isset($v['currency'])) {
								$qb->innerJoin('o.currency', 'currency', Join::WITH, "currency.code = '{$v['currency']}'");
							}

							if (isset($v['lang'])) {
								$vAnd[] = "o.lang = '{$v['lang']}'";
							}

							$or[] = implode(' AND ', $vAnd);
						}
					}

					$qb->andWhere(implode(' OR ', $or));
				} else {
					$qb->andWhere('o.site = :site')
						->setParameter('site', $value);
				}
			});
		}

		if (EshopOrdersConfig::load('invoice.enable') && EshopOrdersConfig::load('ordersGrid.showInvoiceDueDate')) {
			$grid->addColumnText('invoiceDueDate', 'eshopOrders.ordersGrid.invoiceDueDate')
				->setRenderer(function(Order $o) {
					if ($o->getInvoice() === null) {
						return '';
					}

					$now     = (new DateTime)->setTime(0, 0);
					$dueDate = (clone $o->getInvoice()->dueDate)->setTime(0, 0);

					return Html::el('span', ['class' => $now->getTimestamp() > $dueDate->getTimestamp() && !$o->isPaid ? 'text-danger' : ''])
						->setText($dueDate->format('d.m.Y'));
				});
		}

		// Filter
		$grid->addFilterText('products', 'eshopOrders.ordersGrid.searchInProducts')
			->setCondition(function(QueryBuilder $qb, $value) {
				$expr  = $this->em->getExpressionBuilder();
				$subQb = $this->em->getRepository(OrderItem::class)->createQueryBuilder('oi')
					->select('IDENTITY(oi.order)')
					->innerJoin('oi.orderItemTexts', 'oit')
					->andWhere('oi.code1 LIKE :productsL OR oit.name LIKE :productsL');

				$qb->andWhere($expr->in('o.id', $subQb->getDQL()))
					->setParameter('productsL', "%{$value}%");
			});
		$grid->addFilterText('packageNumber', 'eshopOrders.ordersGrid.searchPackageNumber')->setCondition(function(QueryBuilder $qb, $value) {
			$criteria = Criteria::create();
			$this->eventDispatcher->dispatch(new GridFilterEvent($qb, $criteria, $value), OrdersGrid::class . '::filterPackageNumber');
			$qb->addCriteria($criteria);
		});
		$grid->addFilterText('id', '')->setCondition(function(QueryBuilder $qb, $value) {
			$criteria = Criteria::create();
			$criteria->orWhere(Criteria::expr()->eq('o.id', $value));

			$this->eventDispatcher->dispatch(new GridFilterEvent($qb, $criteria, $value), OrdersGrid::class . '::filterId');
			$qb->addCriteria($criteria);
		});

		$speditionItems = ['' => ''] + $this->speditionsService->getForSelectOption();
		$this->eventDispatcher->dispatch(new Event(['items' => &$speditionItems]), OrdersGrid::class . '::onSpeditionItemsLoad');
		$grid->addFilterSelect('spedition', '', $speditionItems, 'spedition.spedition')->setCondition(function(QueryBuilder $qb, $value) {
			if (empty($value)) {
				return;
			}

			$qb->innerJoin('o.spedition', 'oSpedition');

			$criteria = Criteria::create();
			$criteria->orWhere(Criteria::expr()->eq('oSpedition.spedition', $value));
			$this->eventDispatcher->dispatch(new GridFilterEvent($qb, $criteria, $value), OrdersGrid::class . '::filterSpedition');
			$qb->addCriteria($criteria);
		});

		$grid->addFilterDateRange('created', '')->setCondition(function(QueryBuilder $qb, $values) {
			if ($values->from || $values->to) {
				if (!in_array('oStatuses', $qb->getAllAliases()))
					$qb->innerJoin('o.orderStatuses', 'oStatuses');
				$qb->andWhere('oStatuses.status = :statusName2')->setParameter('statusName2', 'created');
			}

			if ($values->from) {
				$date = DateTime::createFromFormat('j. n. Y', $values->from);
				if ($date instanceof \DateTimeInterface) {
					$qb->andWhere('oStatuses.created >= :statusFrom')
						->setParameter('statusFrom', $date->setTime(0, 0, 0));
				}
			}
			if ($values->to) {
				$date = DateTime::createFromFormat('j. n. Y', $values->to);
				if ($date instanceof \DateTimeInterface) {
					$qb->andWhere('oStatuses.created <= :statusTo')
						->setParameter('statusTo', $date->setTime(23, 59, 59));
				}
			}
		});

		if (isset($grid->getColumns()['status'])) {
			$grid->addFilterSelect('status', '', ['' => ''] + $this->statusesService->getForSelectOption())->setCondition(function(QueryBuilder $qb, $value) {
				$join = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('os2')
					->select('GROUP_CONCAT(IDENTITY(os2.status))')
					->andWhere('os2.order = o.id')
					->andWhere('os2.deleted IS NULL OR os2.deleted = 0')
					->groupBy('os2.order')
					->orderBy('os2.order', 'DESC')
					->getQuery();

				$qb->addSelect("(" . $join->getDQL() . ") as hidden statusFilter")
					->andHaving('statusFilter LIKE :status2')
					->setParameter('status2', "%$value");
			});
			$grid->getColumn('status')->getElementPrototype('th')->addClass('w1nw');
		}

		$grid->setHeadFilters(['products', 'packageNumber']);

		// ItemDetail
		$grid->setItemsDetail($this->getTemplateFile('OrdersGridDetail', 'Order'));

		// Actions
		$grid->addAction('edit', '', 'Default:editOrder')->setIcon('edit')->setBsType('primary');
		if (EshopOrdersConfig::load('orderForm.allowOrderDelete')) {
			$grid->addAction('delete', '', 'delete!')
				->setConfirmation(
					new StringConfirmation('eshopOrders.orderForm.reallyDelete')
				)->setIcon('times')
				->setBsType('danger');
		}

		$grid->addGroupSelectCustomAction('eshopOrders.defaultGrid.changeStatus', $this->statusesService->getForSelectOption() + [
				OrderStatus::STATUS_IS_PAID => 'eshopOrders.ordersGrid.statusesLong.isPaid',
			])->onSelect[] = [$this, 'gridChangeStatus'];

		foreach ($this->exportCallbacks as $k => $v) {
			if (EshopOrdersConfig::load('expeditionOrdersGrid.enableAdvancedOptions', false) && Strings::contains($k, 'ceskaposta') && !Strings::endsWith($k, 'Labels')) {
				$grid->addGroupAction($v['title'])->onSelect[] = $v['callback'];
			} else if (!Strings::endsWith($k, 'Labels')) {
				$grid->addGroupTextAction($v['title'])
					->setAttribute('placeholder', $this->t('eshopOrders.expeditionOrdersGrid.form.quantity'))
					->onSelect[] = $v['callback'];
			} else {
				$grid->addGroupAction($v['title'])->onSelect[] = $v['callback'];
			}
		}

		$grid->addAction('pdf', '')->setRenderer(function(Order $order) {
			if ($order->getInvoice() === null) {
				return '';
			}

			return Html::el('a', ['href'  => $this->presenter->link('Orders:invoice', $order->getId()),
			                      'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
				->addHtml(Html::el('i', ['class' => 'fas fa-file-invoice']));
		});

		$grid->addGroupAction('eshopOrders.ordersGrid.printOrders')->onSelect[] = function(array $ids) {
			$presenter                          = $this->getPresenter();
			$presenter->payload->fileRequests[] = [
				'name' => 'print',
				'url'  => $presenter->link(':EshopOrders:Admin:Orders:print', ['ids' => $ids]),
			];

			$presenter->sendPayload();
		};

		if (EshopOrdersConfig::load('enableDeliveryListPrint')) {
			$grid->addGroupAction('eshopOrders.ordersGrid.printDeliveryList')->onSelect[] = function(array $ids) {
				$presenter                          = $this->getPresenter();
				$presenter->payload->fileRequests[] = [
					'name' => 'print',
					'url'  => $presenter->link(':EshopOrders:Admin:Orders:printDeliveryList', ['ids' => $ids]),
				];

				$presenter->sendPayload();
			};

			foreach ($this->langsService->getLangs(false) as $lang) {
				$grid->addGroupAction($this->t('eshopOrders.ordersGrid.printDeliveryListLang', [
					'lang' => $lang->getShortTitle(),
				]))
					->onSelect[] = function(array $ids) use ($lang) {
					$presenter                          = $this->getPresenter();
					$presenter->payload->fileRequests[] = [
						'name' => 'print',
						'url'  => $presenter->link(':EshopOrders:Admin:Orders:printDeliveryList', [
							'ids'  => $ids,
							'lang' => $lang->getTag(),
						]),
					];

					$presenter->sendPayload();
				};
			}
		}

		$preorderAv = $this->availabilityService->getAllByIdent()[Availability::PREORDER] ?? null;
		// Render condition
		$grid->setRowCallback(function(Order $order, Html $tr) use (&$orderItems, $preorderAv) {
			if (EshopOrdersConfig::load('customer.showBlacklisted', false) && $order->getCustomer() && $order->getCustomer()->isBlacklisted) {
				$tr->addClass('bg-warning');
			}

			$event = new Event(['html' => $tr, 'order' => $order]);
			$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::onRowCallback');

			if (!$preorderAv)
				return;

			foreach ($orderItems[$order->getId()] as $item) {
				if ($item->getMoreDataValue('isPreorder', false)) {
					$tr->addClass('av--preorder');
					break;
				}
			}
		});

		// Columns prototype
		$grid->getColumn('id')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('created')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('phone')->getElementPrototype('td')->addClass('w1nw');

		return $grid;
	}

	protected function createComponentExpeditionAdvancedOptionsForm()
	{
		return $this->advancedOptionsFormFactory->create();
	}

	protected function createComponentOrderMessageModal()
	{
		return $this->orderMessageModalFactory->create();
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if (EshopOrdersConfig::load('orderForm.allowOrderDelete')) {
			if ($this->ordersService->remove($id)) {
				$presenter->flashMessageSuccess('default.removed');
			} else {
				$presenter->flashMessageDanger('default.removeFailed');
			}
		} else {
			$presenter->flashMessageDanger('eshopOrders.orderForm.removeDenied');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridChangeStatus(array $ids, string $value): void
	{
		if (!$value || empty($ids))
			return;

		$error     = true;
		$presenter = $this->getPresenter();

		foreach ($ids as $orderId) {
			$order = $this->ordersService->getReference($orderId);
			if (!$order) {
				continue;
			}
			$event            = new OrderStatusEvent($order, $value);
			$event->presenter = $presenter;
			$this->eventDispatcher->dispatch($event, OrdersGrid::class . '::changeStatus');
		}

		if ($value === OrderStatus::STATUS_IS_PAID) {
			AppState::setState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.manual');
			if ($this->ordersService->setPaid($ids))
				$error = false;
		} else if ($this->statusesService->changeStatus($ids, $value))
			$error = false;

		if ($error)
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		else
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');
		}
	}

	public function handleOrderMessageModal(int $orderId): void
	{
		$this['orderMessageModal']->setOrder($orderId);
		$this->template->modal      = 'orderMessageModal';
		$this->template->modalTitle = $this->t('eshopOrders.orderMessageModal.title', null, ['order' => $orderId]);
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * =================  GET SET
	 */

	public function addExportCallback(string $key, string $title, callable $callback): self
	{
		$this->exportCallbacks[$key] = [
			'title'    => $title,
			'callback' => $callback,
		];

		return $this;
	}

	/*******************************************************************************************************************
	 * =================  OTHERS
	 */

	public function openExpeditionAdvancedOptionsFormModal(array $orderIds, string $speditionIdent): void
	{
		$this['expeditionAdvancedOptionsForm']->setDefaults($orderIds, $speditionIdent);
		$this->template->modal      = 'expeditionAdvancedOptionsForm';
		$this->template->modalTitle = $this->t('eshopOrders.expeditionAdvancedOptionsForm.title');
		$this->redrawControl('modal');
	}

	public function getAddress(?OrderAddress $oa): ?OrderAddress
	{
		return $oa ? $this->cAddresses[$oa->getId()] : null;
	}

	public function joinFilterAddressInvoice(QueryBuilder $qb): void
	{
		if (!Arrays::contains($qb->getAllAliases(), 'ai')) {
			$qb->leftJoin('o.addressInvoice', 'ai');
		}
	}

	public function joinFilterAddressDelivery(QueryBuilder $qb): void
	{
		if (!Arrays::contains($qb->getAllAliases(), 'ad')) {
			$qb->leftJoin('o.addressDelivery', 'ad');
		}
	}
}
