<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\OrderCurrency;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Exchange;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderCurrency;
use EshopOrders\Model\Invoices;
use Nette\Utils\ArrayHash;

class OrderCurrencyForm extends BaseControl
{
	protected Order      $order;
	protected Currencies $currencies;
	protected Exchange   $exchange;
	protected Invoices   $invoices;

	protected ?array $cCurrencies = null;

	public function __construct(
		Order      $order,
		Currencies $currencies,
		Exchange   $exchange,
		Invoices   $invoices
	)
	{
		$this->order      = $order;
		$this->currencies = $currencies;
		$this->exchange   = $exchange;
		$this->invoices   = $invoices;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);
		$form->setAjax();

		$form->addSelect('currency', 'eshopOrders.order.currency', $this->getCurrencies())
			->setDefaultValue($this->order->currency->code ?? $this->currencies->getDefaultCode())
			->setRequired();
		$form->addSubmit('save', 'default.save')
			->setHtmlAttribute('data-confirm', $this->t('default.reallyConfirm'));

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$currency = $this->order->currency;

			if ($values->currency === $this->currencies->getDefaultCode()) {
				if ($currency) {
					$this->em->remove($currency);
					$this->order->currency = null;
				}
			} else {
				if (!$currency) {
					$currency = new OrderCurrency($this->order, $values->currency);

					$this->order->currency = $currency;
				}

				/** @var Currency|null $currencyEntity */
				$currencyEntity = $this->currencies->getEr()->findOneBy(['code' => $values->currency, 'site' => [null, $this->order->site->getIdent()]]);

				$rate           = $this->exchange->changeByDate(1, $this->order->getCreatedTime(), $this->currencies->getDefaultCode(), $values->currency);
				$currency->rate = Strings::formatEntityDecimal($rate);

				if ($currencyEntity) {
					$currency->decimals = $currencyEntity->decimals;
				}

				$this->em->persist($currency);
			}

			$this->em->flush();

			if ($this->order->getInvoice()) {
				$this->invoices->regenerateInvoice($this->order->getInvoice()->getId());
			}

			$this->presenter->redirect('this');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}
	}

	protected function getCurrencies(): array
	{
		if ($this->cCurrencies === null) {
			$this->cCurrencies = [];

			foreach ($this->currencies->getEr()->findAll() as $row) {
				/** @var Currency $row */
				if (
					$row->isActive
					&& (!$row->site || $row->site->getIdent() === $this->order->site->getIdent())
				) {
					$this->cCurrencies[$row->getCode()] = $row->getCode();
				}
			}
		}

		return $this->cCurrencies;
	}
}
