<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\PaymentSpedition;

use Core\AdminModule\Model\Sites;
use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Currency\Model\Currencies;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Payments;
use EshopOrders\Model\PaymentSpeditions;
use Core\Model\Templating\Filters\Price as PriceFilter;

class PaymentSpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public ?int $id = null;

	protected PaymentSpeditions $paymentSpeditionsService;
	public PaymentSpedition     $paymentSpedition;
	protected Speditions        $speditionsService;
	protected Payments          $paymentsService;
	protected Sites             $sitesService;
	protected Countries         $countriesService;
	protected Currencies        $currencies;
	protected GroupsCustomers   $groupsCustomers;
	protected PriceFilter       $priceFilter;
	protected EshopOrdersCache  $eshopOrdersCache;

	public function __construct(
		PaymentSpeditions $paymentSpeditions,
		Speditions        $speditions, Payments $payments,
		Sites             $sites,
		Countries         $countries,
		Currencies        $currencies,
		PriceFilter       $priceFilter,
		GroupsCustomers   $groupsCustomers,
		EshopOrdersCache  $eshopOrdersCache
	)
	{
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->sitesService             = $sites;
		$this->countriesService         = $countries;
		$this->currencies               = $currencies;
		$this->priceFilter              = $priceFilter;
		$this->groupsCustomers          = $groupsCustomers;
		$this->eshopOrdersCache         = $eshopOrdersCache;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$paymentChoices = $speditionChoices = [];
		$speditions     = $this->speditionsService->getAll();
		$payments       = $this->paymentsService->getAll();
		$sites          = $this->sitesService->getAll();

		foreach ($speditions as $s) {
			$price                         = $this->priceFilter->format((float) $s->getPrice());
			$speditionChoices[$s->getId()] = $s->getName() . ' - ' . $price;
		}

		foreach ($payments as $p) {
			$price                       = $this->priceFilter->format((float) $p->getPrice());
			$paymentChoices[$p->getId()] = $p->getName() . ' - ' . $price;
		}

		$form->addSelect('spedition', 'eshopOrders.orderPage.spedition', $speditionChoices)
			->setTranslator(null)
			->setRequired();
		$form->addSelect('payment', 'eshopOrders.orderPage.payment', $paymentChoices)
			->setTranslator(null)
			->setRequired();
		if (count($sites) > 1)
			$form->addCheckboxList('sites', 'eshopOrders.paymentSpedition.sites', $this->sitesService->getOptionsForSelect())
				->setTranslator(null)
				->setRequired();
		else
			$form->addHidden('site', $sites[0]->getIdent());

		$form->addCheckboxList('countries', 'eshopOrders.paymentSpedition.country', $this->countriesService->getAllNameColumn())
			->setTranslator(null);

		if (EshopOrdersConfig::load('paymentSpeditions.allowCustomerGroups', false)) {
			$form->addCheckboxList('customerGroups', 'eshopOrders.paymentSpedition.customerGroups', $this->groupsCustomers->getOptionsForSelect());
			$form->addCheckbox('customerWithoutGroup', 'eshopOrders.paymentSpedition.customerWithoutGroup')
				->setDefaultValue(0);
			$form->addCheckbox('noCustomer', 'eshopOrders.paymentSpedition.noCustomer')
				->setDefaultValue(0);
		}

		$currencies = ['' => ''];
		foreach ($this->currencies->getActive() as $c)
			$currencies[$c->getCode()] = $c->getCode();
		$form->addSelect('currency', 'eshopOrders.paymentSpedition.currency', $currencies);

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$form->addDateTimePicker('freeFromDate', 'eshopOrders.paymentSpedition.freeFromDate');
			$form->addDateTimePicker('freeToDate', 'eshopOrders.paymentSpedition.freeToDate');
		}

		$form->addBool('isPublished', 'default.isActive')->setDefaultValue(1);
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$paymentSpedition = $this->paymentSpeditionsService->get($this->id);
				$flashMessage     = 'eshopOrders.paymentSpeditionForm.edited';
			} else {
				$paymentSpedition = new PaymentSpedition();
				$flashMessage     = 'eshopOrders.paymentSpeditionForm.added';
			}

			// Sites
			$sites     = $paymentSpedition->sites->toArray();
			$formSites = $values->site ? [$values->site] : $values->sites;

			foreach (array_diff(array_keys($sites), $formSites) as $v) {
				$paymentSpedition->sites->remove($v);
			}
			foreach (array_diff($formSites, array_keys($sites)) as $v) {
				$paymentSpedition->sites->set($v, $this->sitesService->getReference($v));
			}

			// Countries
			$countries = $paymentSpedition->countries->toArray();
			foreach (array_diff(array_keys($countries), $values->countries) as $v) {
				$paymentSpedition->countries->remove($v);
			}
			foreach (array_diff($values->countries, array_keys($countries)) as $v) {
				$paymentSpedition->countries->set($v, $this->countriesService->getReference($v));
			}

			// Customer groups
			if (EshopOrdersConfig::load('paymentSpeditions.allowCustomerGroups', false)) {
				$customerGroups = $paymentSpedition->customerGroups->toArray();
				foreach (array_diff(array_keys($customerGroups), $values->customerGroups) as $v) {
					$paymentSpedition->customerGroups->remove($v);
				}
				foreach (array_diff($values->customerGroups, array_keys($customerGroups)) as $v) {
					$paymentSpedition->customerGroups->set($v, $this->groupsCustomers->getReference($v));
				}

				$paymentSpedition->customerWithoutGroup = (int) $values->customerWithoutGroup;
				$paymentSpedition->noCustomer           = (int) $values->noCustomer;
			}

			$paymentSpedition->setPayment($this->paymentsService->get($values->payment));
			$paymentSpedition->setSpedition($this->speditionsService->get($values->spedition));
			$paymentSpedition->isPublished = $values->isPublished;
			if (EshopOrdersConfig::load('allowFreeFrom')) {
				$paymentSpedition->freeFromDate = $values->freeFromDate;
				$paymentSpedition->freeToDate   = $values->freeToDate;
			}
			$paymentSpedition->currency = $values->currency ?: null;

			$this->em->persist($paymentSpedition)->flush();
			$form->addCustomData('paymentSpeditionId', $paymentSpedition->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['spedition', 'payment', 'freeFrom'],
			]);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setPaymentSpedition($id): void
	{
		$this->id               = $id;
		$this->paymentSpedition = $this->paymentSpeditionsService->get($id);

		if (!$this->paymentSpedition)
			throw new InvalidArgumentException();

		$m = $this->paymentSpedition;
		$f = $this['form'];
		$d = [
			'payment'      => $m->getPayment()->getId(),
			'spedition'    => $m->getSpedition()->getId(),
			'isPublished'  => $m->isPublished,
			'freeFromDate' => $m->freeFromDate,
			'freeToDate'   => $m->freeToDate,
		];

		if ($f->getComponent('site', false))
			$d['site'] = $m->sites->first() ? $m->sites->first()->getIdent() : null;
		else
			foreach ($m->sites->toArray() as $site)
				if (array_key_exists($site->getIdent(), $f['sites']->getItems()))
					$d['sites'][] = $site->getIdent();

		foreach ($m->countries->toArray() as $country)
			if (array_key_exists($country->getId(), $f['countries']->getItems()))
				$d['countries'][] = $country->getId();

		if (EshopOrdersConfig::load('paymentSpeditions.allowCustomerGroups', false)) {
			foreach ($m->customerGroups as $cg) {
				if (isset($f['customerGroups']->getItems()[$cg->getId()])) {
					$d['customerGroups'][] = $cg->getId();
				}
			}

			$d['customerWithoutGroup'] = $m->customerWithoutGroup;
			$d['noCustomer']           = $m->noCustomer;
		}

		if ($m->currency && array_key_exists($m->currency, $f['currency']->getItems()))
			$d['currency'] = $m->currency;

		$f->setDefaults($d);
	}
}
